//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/src/DwmDnsRRDataCERT.cc 10133 $
// @(#) $Id: DwmDnsRRDataCERT.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataCERT.cc
//!  \brief Dwm::Dns::RRDataCERT class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataCERT.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/src/DwmDnsRRDataCERT.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataCERT::RRDataCERT()
        : _type(0), _keyTag(0), _algorithm(0), _certificate()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataCERT::RRDataCERT(uint16_t type, uint16_t keyTag, uint8_t algorithm,
                         const string & certificate)
        : _type(type), _keyTag(keyTag), _algorithm(algorithm),
          _certificate(certificate)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataCERT::operator == (const RRDataCERT & cert) const
    {
      bool  rc = false;
      if ((_type == cert._type)
          && (_keyTag == cert._keyTag)
          && (_algorithm == cert._algorithm)
          && (_certificate == cert._certificate)) {
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataCERT::Type() const
    {
      return _type;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataCERT::Type(uint16_t type)
    {
      _type = type;
      return _type;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataCERT::KeyTag() const
    {
      return _keyTag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataCERT::KeyTag(uint16_t keyTag)
    {
      _keyTag = keyTag;
      return _keyTag;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataCERT::Algorithm() const
    {
      return _algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataCERT::Algorithm(uint8_t algorithm)
    {
      _algorithm = algorithm;
      return _algorithm;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataCERT::Certificate() const
    {
      return _certificate;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataCERT::Certificate(const string & certificate)
    {
      _certificate = certificate;
      return _certificate;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataCERT::Encode(uint8_t *pkt, uint8_t *ptr,
                                uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 5) <= (pkt + pktlen)) {
        uint16_t  u16val = htons(_type);
        memcpy(ptr, &u16val, sizeof(u16val));
        ptr += sizeof(u16val);
        u16val = htons(_keyTag);
        memcpy(ptr, &u16val, sizeof(u16val));
        ptr += sizeof(u16val);
        *ptr++ = _algorithm;
        if (_certificate.size() > 0) {
          if ((ptr + _certificate.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _certificate.c_str(), _certificate.size());
            ptr += _certificate.size();
          }
          else {
            throw out_of_range("Dwn::Dns::RRDataKCERT will not fit in packet");
          }
        }
      }
      else {
        throw out_of_range("Dwn::Dns::RRDataKCERT will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataCERT::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                     uint16_t pktlen, uint16_t rdlen)
    {
      _type = 0;
      _keyTag = 0;
      _algorithm = 0;
      _certificate.clear();
      
      uint16_t        bytesLeft = rdlen;
      const uint8_t  *origptr = ptr;
      
      if ((ptr + 5) <= (pkt + pktlen)) {
        uint16_t  u16val;
        memcpy(&u16val, ptr, sizeof(u16val));
        ptr += sizeof(u16val);
        _type = ntohs(u16val);
        memcpy(&u16val, ptr, sizeof(u16val));
        ptr += sizeof(u16val);
        _keyTag = ntohs(u16val);
        _algorithm = *ptr++;
        bytesLeft -= (ptr - origptr);
        if (bytesLeft) {
          _certificate.assign((const char *)ptr, bytesLeft);
          ptr += bytesLeft;
        }
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataCERT & cert)
    {
      if (os) {
        os << cert._type << ' ' << cert._keyTag << ' '
           << (uint16_t)cert._algorithm << ' '
           << Base64::Encode(cert._certificate);
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
