//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.6/classes/include/DwmDnsLabelSequence.hh 10085 $
// @(#) $Id: DwmDnsLabelSequence.hh 10085 2018-01-26 04:22:25Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsLabelSequence.hh
//!  \brief Dwm::Dns::LabelSequence class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSLABELSEQUENCE_HH_
#define _DWMDNSLABELSEQUENCE_HH_

#include <cstdint>
#include <map>
#include <string>
#include <vector>

#include "DwmDnsLabelPositions.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  From RFC1035 section 4.1.4:
    //!    The compression scheme allows a domain name in a message to be
    //!    represented as either:
    //!      - a sequence of labels ending in a zero octet
    //!      - a pointer
    //!      - a sequence of labels ending with a pointer
    //!
    //!  What this means is that a domain name isn't standalone, since it
    //!  can contain a pointer to elsewhere.  Of course this completely
    //!  defeats encapsulation.
    //!  I only need the compression facilities when encoding packets.
    //!  To facilitate encoding, I use an instance of LabelPositions to
    //!  keep track of potential pointer targets.
    //------------------------------------------------------------------------
    class LabelSequence
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructs an empty label sequence.
      //----------------------------------------------------------------------
      LabelSequence();

      //----------------------------------------------------------------------
      //!  Construct a label sequence from given string @c value.
      //----------------------------------------------------------------------
      LabelSequence(const std::string & value);

      //----------------------------------------------------------------------
      //!  Convert to a string.
      //----------------------------------------------------------------------
      operator std::string () const;
      
      //----------------------------------------------------------------------
      //!  Encodes the label sequence into buffer @ pkt of length @c pktlen,
      //!  starting at @c ptr.  Returns the address immediately following
      //!  the encoded label sequence in @c pkt on success.  Throws
      //!  std::out_of_range if @c pkt is too short to contain the encoded
      //!  label sequence.  @c lps is used for label compression as documented
      //!  in section 4.1.4 of RFC1035.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Encode without compression.  Used for NSEC records, for example.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen) const;

      //----------------------------------------------------------------------
      //!  Decodes the label sequence from buffer @c pkt of length @c pktlen,
      //!  starting at @c ptr.  Returns the address immediately following
      //!  the encoded label sequence in @c pkt on success.  Throws
      //!  std::out_of_range if @c pkt was too short to contain an encoded
      //!  label sequence.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen);

    protected:
      std::string  _value;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSLABELSEQUENCE_HH_
