//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataNSEC3.cc 10133 $
// @(#) $Id: DwmDnsRRDataNSEC3.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataNSEC3.cc
//!  \brief Dwm::Dns::RRDataNSEC3 class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataNSEC3.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataNSEC3.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataNSEC3::RRDataNSEC3()
        : _hashAlgorithm(0), _flags(0), _iterations(0), _salt(),
          _hashedOwnerName(), _bitmaps()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataNSEC3::RRDataNSEC3(uint8_t hashAlgorithm, uint8_t flags,
                             uint16_t iterations, const string & salt,
                             const string & hashedOwnerName,
                             const std::vector<NSECBitmap> & bitmaps)
        : _hashAlgorithm(hashAlgorithm), _flags(flags),
          _iterations(iterations), _salt(salt),
          _hashedOwnerName(hashedOwnerName), _bitmaps(bitmaps)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataNSEC3::operator == (const RRDataNSEC3 & nsec3) const
    {
      return ((_hashAlgorithm == nsec3._hashAlgorithm)
              && (_flags == nsec3._flags)
              && (_iterations == nsec3._iterations)
              && (_salt == nsec3._salt)
              && (_hashedOwnerName == nsec3._hashedOwnerName)
              && (_bitmaps == nsec3._bitmaps));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3::HashAlgorithm() const
    {
      return _hashAlgorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3::HashAlgorithm(uint8_t hashAlgorithm)
    {
      _hashAlgorithm = hashAlgorithm;
      return _hashAlgorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3::Flags() const
    {
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3::Flags(uint8_t flags)
    {
      _flags = flags;
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataNSEC3::Iterations() const
    {
      return _iterations;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataNSEC3::Iterations(uint16_t iterations)
    {
      _iterations = iterations;
      return _iterations;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataNSEC3::Salt() const
    {
      return _salt;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataNSEC3::Salt(const string & salt)
    {
      _salt = salt;
      return _salt;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataNSEC3::HashedOwnerName() const
    {
      return _hashedOwnerName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string &
    RRDataNSEC3::HashedOwnerName(const string & hashedOwnerName)
    {
      _hashedOwnerName = hashedOwnerName;
      return _hashedOwnerName;
    }

    //------------------------------------------------------------------------
    //!                                                                       
    //------------------------------------------------------------------------
    const vector<NSECBitmap> & RRDataNSEC3::Bitmaps() const
    {
      return _bitmaps;
    }
    
    //------------------------------------------------------------------------
    //!                                                                       
    //------------------------------------------------------------------------
    const vector<NSECBitmap> &
    RRDataNSEC3::Bitmaps(const vector<NSECBitmap> & bitmaps)
    {
      _bitmaps = bitmaps;
      return _bitmaps;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataNSEC3::Encode(uint8_t *pkt, uint8_t *ptr,
                                 uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 6) <= (pkt + pktlen)) {
        *ptr++ = _hashAlgorithm;
        *ptr++ = _flags;
        uint16_t  val = htons(_iterations);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        *ptr++ = (uint8_t)(_salt.size() & 0xFF);
        if (! _salt.empty()) {
          if ((ptr + _salt.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _salt.c_str(), _salt.size());
            ptr += _salt.size();
          }
          else {
            throw out_of_range("Dwm::Dns::RRDataNSEC3 will not fit in packet");
          }
        }
        *ptr++ = (uint8_t)(_hashedOwnerName.size() & 0xFF);
        if (! _hashedOwnerName.empty()) {
          if ((ptr + _hashedOwnerName.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _hashedOwnerName.c_str(), _hashedOwnerName.size());
            ptr += _hashedOwnerName.size();
          }
          else {
            throw out_of_range("Dwm::Dns::RRDataNSEC3 will not fit in packet");
          }
        }
        for (auto & bitmap : _bitmaps) {
          ptr = bitmap.Encode(pkt, ptr, pktlen);
        }
      }
      else {
        throw out_of_range("Dwm::Dns::RRDataNSEC3 will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataNSEC3::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                       uint16_t pktlen, uint16_t rdlen)
    {
      _hashAlgorithm = 0;
      _flags = 0;
      _iterations = 0;
      _salt.clear();
      _hashedOwnerName.clear();
      _bitmaps.clear();

      const uint8_t  *origptr = ptr;
      int             bytesRemaining = rdlen;

      if ((ptr + 6) <= (pkt + pktlen)) {
        _hashAlgorithm = *ptr++;
        _flags = *ptr++;
        memcpy(&_iterations, ptr, sizeof(_iterations));
        _iterations = ntohs(_iterations);
        ptr += sizeof(_iterations);
        uint8_t  len = *ptr++;
        if ((ptr + len) <= (pkt + pktlen)) {
          _salt.assign((const char *)ptr, len);
          ptr += len;
        }
        else {
          throw out_of_range("packet too short to contain"
                             " Dwm::Dns::RRDataNSEC3");
        }
        if (ptr <= (pkt + pktlen)) {
          len = *ptr++;
          if ((ptr + len) <= (pkt + pktlen)) {
            _hashedOwnerName.assign((const char *)ptr, len);
            ptr += len;
          }
          else {
            throw	out_of_range("packet too short to contain"
                               " Dwm::Dns::RRDataNSEC3");
          }
          bytesRemaining -= (ptr - origptr);
          while (bytesRemaining > 0) {
            const uint8_t  *bmptr = ptr;
            NSECBitmap  bitmap;
            ptr = bitmap.Decode(pkt, ptr, pktlen, bytesRemaining);
            bytesRemaining -= (ptr - bmptr);
            _bitmaps.push_back(bitmap);
          }
        }
        else {
          throw out_of_range("packet too short to contain"
                             " Dwm::Dns::RRDataNSEC3");
        }
      }
      else {
        throw out_of_range("packet too short to contain"
                           " Dwm::Dns::RRDataNSEC3");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const	RRDataNSEC3 & nsec3)
    {
      if (os) {
        os << (uint16_t)nsec3._hashAlgorithm << ' '
           << (uint16_t)nsec3._flags << ' ' << nsec3._iterations << ' '
           << Base64::Encode(nsec3._salt) << ' '
           << Base64::Encode(nsec3._hashedOwnerName);
        for (auto & bm : nsec3._bitmaps) {
          os << bm;
        }
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
