//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataKEY.cc 10133 $
// @(#) $Id: DwmDnsRRDataKEY.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataKEY.cc
//!  \brief Dwm::Dns::RRDataKEY class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <iomanip>
#include <sstream>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataKEY.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsRRDataKEY.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataKEY::RRDataKEY()
        : _flags(0), _protocol(0), _algorithm(0), _moreFlags(0), _publicKey()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataKEY::RRDataKEY(uint16_t flags, uint8_t protocol, uint8_t algorithm,
                         uint16_t moreFlags, const string & publicKey)
        : _flags(flags), _protocol(protocol), _algorithm(algorithm),
          _moreFlags(moreFlags), _publicKey(publicKey)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataKEY::operator == (const RRDataKEY & key) const
    {
      bool  rc = false;
      if ((_flags == key._flags)
          && (_protocol == key._protocol)
          && (_algorithm == key._algorithm)
          && (_publicKey == key._publicKey)) {
        if (_flags & k_flagsXTMask) {
          if (_moreFlags == key._moreFlags) {
            rc = true;
          }
        }
        else {
          rc = true;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataKEY::Flags() const
    {
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataKEY::Flags(uint16_t flags)
    {
      if ((flags & k_flagsXTMask) == 0) {
        _moreFlags = 0;
      }
      _flags = flags;
      return _flags;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataKEY::Protocol() const
    {
      return _protocol;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataKEY::Protocol(uint8_t protocol)
    {
      _protocol = protocol;
      return _protocol;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataKEY::Algorithm() const
    {
      return _algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataKEY::Algorithm(uint8_t algorithm)
    {
      _algorithm = algorithm;
      return _algorithm;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataKEY::MoreFlags() const
    {
      return _moreFlags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataKEY::MoreFlags(uint16_t moreFlags)
    {
      if (moreFlags) {
        _flags |= k_flagsXTMask;
      }
      else {
        _flags &= ~(k_flagsXTMask);
      }
      _moreFlags = moreFlags;
      return _moreFlags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataKEY::PublicKey() const
    {
      return _publicKey;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataKEY::PublicKey(const string & publicKey)
    {
      _publicKey = publicKey;
      return _publicKey;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataKEY::Encode(uint8_t *pkt, uint8_t *ptr,
                               uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 4) <= (pkt + pktlen)) {
        uint16_t  flags = htons(_flags);
        memcpy(ptr, &flags, sizeof(flags));
        ptr += sizeof(flags);
        *ptr++ = _protocol;
        *ptr++ = _algorithm;
        if (_flags & k_flagsXTMask) {
          flags = htons(_moreFlags);
          memcpy(ptr, &flags, sizeof(flags));
          ptr += sizeof(flags);
        }
        if (_publicKey.size() > 0) {
          if ((ptr + _publicKey.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _publicKey.c_str(), _publicKey.size());
            ptr += _publicKey.size();
          }
          else {
            throw out_of_range("Dwn::Dns::RRDataKEY will not fit in packet");
          }
        }
      }
      else {
        throw out_of_range("Dwn::Dns::RRDataKEY will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataKEY::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                     uint16_t pktlen, uint16_t rdlen)
    {
      _flags = 0;
      _protocol = 0;
      _algorithm = 0;
      _moreFlags = 0;
      _publicKey.clear();
      
      int             bytesLeft = rdlen;
      const uint8_t  *origptr = ptr;
      
      if ((ptr + 4) <= (pkt + pktlen)) {
        uint16_t  flags;
        memcpy(&flags, ptr, sizeof(flags));
        ptr += sizeof(flags);
        _flags = ntohs(flags);
        _protocol = *ptr++;
        _algorithm = *ptr++;
        if (_flags & k_flagsXTMask) {
          memcpy(&flags, ptr, sizeof(flags));
          ptr += sizeof(flags);
          _moreFlags = ntohs(flags);
        }
        bytesLeft -= (ptr - origptr);
        if (bytesLeft > 0) {
          _publicKey.assign((const char *)ptr, bytesLeft);
          ptr += bytesLeft;
        }
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream &	os, const RRDataKEY & key)
    {
      if (os) {
        ostringstream  oss;
        oss << setfill('0')
            << hex << "0x" << setw(4) << key._flags << ' '
            << dec << (uint16_t)key._protocol << ' '
            << (uint16_t)key._algorithm << ' ';
        if (key._flags & RRDataKEY::k_flagsXTMask) {
          oss << hex << "0x" << setw(4) << key._moreFlags << dec << ' ';
        }
        os << oss.str() << ' ' << Base64::Encode(key._publicKey);
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
