//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsMessageHeader.cc 10133 $
// @(#) $Id: DwmDnsMessageHeader.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsMessageHeader.cc
//!  \brief Dwm::Dns::MessageHeader class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsMessageHeader.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/src/DwmDnsMessageHeader.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    static const uint16_t  k_headerQRMask     = 0x8000;  // query/response
    static const uint16_t  k_headerOpcodeMask = 0x7800;  // OPCODE
    static const uint16_t  k_headerAAMask     = 0x0400;  // auth. answer
    static const uint16_t  k_headerTCMask     = 0x0200;  // truncated
    static const uint16_t  k_headerRDMask     = 0x0100;  // recursion desired
    static const uint16_t  k_headerRAMask     = 0x0080;  // recursion avail.
    static const uint16_t  k_headerZMask      = 0x0070;  // unused flag bits
    static const uint16_t  k_headerRcodeMask  = 0x000F;  // RCODE
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    MessageHeader::MessageHeader()
    {
      memset(&_data, 0, sizeof(_data));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool
    MessageHeader::operator == (const MessageHeader & messageHeader) const
    {
      return (memcmp(&_data, &messageHeader._data, sizeof(_data)) == 0);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void MessageHeader::Clear()
    {
      memset(&_data, 0, sizeof(_data));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::Id() const
    {
      return _data.id;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::Id(uint16_t id)
    {
      _data.id = id;
      return _data.id;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::IsResponse() const
    {
      return ((_data.flags & k_headerQRMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::IsResponse(bool isResponse)
    {
      if (isResponse) {
        _data.flags |= k_headerQRMask;
      }
      else {
        _data.flags &= ~(k_headerQRMask);
      }
      return ((_data.flags & k_headerQRMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t MessageHeader::OpCode() const
    {
      return ((_data.flags & k_headerOpcodeMask) >> 11);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t MessageHeader::OpCode(uint8_t opcode)
    {
      _data.flags &= ~(k_headerOpcodeMask);
      _data.flags |= ((opcode & 0xF) << 11);
      return ((_data.flags & k_headerOpcodeMask) >> 11);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::IsAuthoritativeAnswer() const
    {
      return ((_data.flags & k_headerAAMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::IsAuthoritativeAnswer(bool isAuthoritative)
    {
      if (isAuthoritative) {
        _data.flags |= k_headerAAMask;
      }
      else {
        _data.flags &= ~(k_headerAAMask);
      }
      return ((_data.flags & k_headerAAMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::IsTruncated() const
    {
      return ((_data.flags & k_headerTCMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::IsTruncated(bool isTruncated)
    {
      if (isTruncated) {
        _data.flags |= k_headerTCMask;
      }
      else {
        _data.flags &= ~(k_headerTCMask);
      }
      return ((_data.flags & k_headerTCMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::RecursionDesired() const
    {
      return ((_data.flags & k_headerRDMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::RecursionDesired(bool recursionDesired)
    {
      if (recursionDesired) {
        _data.flags |= k_headerRDMask;
      }
      else {
        _data.flags &= ~(k_headerRDMask);
      }
      return ((_data.flags & k_headerRDMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::RecursionAvailable() const
    {
      return ((_data.flags & k_headerRAMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageHeader::RecursionAvailable(bool recursionAvailable)
    {
      if (recursionAvailable) {
        _data.flags |= k_headerRAMask;
      }
      else {
        _data.flags &= ~(k_headerRAMask);
      }
      return ((_data.flags & k_headerRAMask) != 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t MessageHeader::Z() const
    {
      return ((_data.flags & k_headerZMask) >> 4);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t MessageHeader::Z(uint8_t z)
    {
      _data.flags &= ~(k_headerZMask);
      _data.flags |= ((z << 4) & k_headerZMask);
      return ((_data.flags & k_headerZMask) >> 4);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t MessageHeader::ResponseCode() const
    {
      return (_data.flags & k_headerRcodeMask);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t MessageHeader::ResponseCode(uint8_t responseCode)
    {
      _data.flags &= ~(k_headerRcodeMask);
      _data.flags |= (responseCode & k_headerRcodeMask);
      return (_data.flags & k_headerRcodeMask);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::QuestionCount() const
    {
      return _data.questionCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::QuestionCount(uint16_t questionCount)
    {
      _data.questionCount = questionCount;
      return _data.questionCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::AnswerCount() const
    {
      return _data.answerCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::AnswerCount(uint16_t answerCount)
    {
      _data.answerCount = answerCount;
      return _data.answerCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::AuthorityCount() const
    {
      return _data.authorityCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::AuthorityCount(uint16_t authorityCount)
    {
      _data.authorityCount = authorityCount;
      return _data.authorityCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::AdditionalCount() const
    {
      return _data.additionalCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageHeader::AdditionalCount(uint16_t additionalCount)
    {
      _data.additionalCount = additionalCount;
      return _data.additionalCount;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *MessageHeader::Encode(uint8_t *pkt, uint8_t *ptr,
                                   uint16_t pktlen) const
    {
      if ((ptr + sizeof(_data)) < (pkt + pktlen)) {
        data_t  data = _data;
        ToNetworkByteOrder(data);
        memcpy(ptr, &data, sizeof(data));
        ptr += sizeof(data);
      }
      else {
        throw out_of_range("Dwm::Dns::MessageHeader will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *MessageHeader::Decode(const uint8_t *pkt,
                                         const uint8_t *ptr,
                                         uint16_t pktlen)
    {
      if ((ptr + sizeof(_data)) < (pkt + pktlen)) {
        memcpy(&_data, ptr, sizeof(_data));
        ToHostByteOrder(_data);
        ptr += sizeof(_data);
      }
      else {
        throw out_of_range("packet too short to contain"
                           " Dwm::Dns::MessageHeader");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void MessageHeader::ToHostByteOrder(data_t & data)
    {
      data.id              = ntohs(data.id);
      data.flags           = ntohs(data.flags);
      data.questionCount   = ntohs(data.questionCount);
      data.answerCount     = ntohs(data.answerCount);
      data.authorityCount  = ntohs(data.authorityCount);
      data.additionalCount = ntohs(data.additionalCount);
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void MessageHeader::ToNetworkByteOrder(data_t & data)
    {
      data.id              = htons(data.id);
      data.flags           = htons(data.flags);
      data.questionCount   = htons(data.questionCount);
      data.answerCount     = htons(data.answerCount);
      data.authorityCount  = htons(data.authorityCount);
      data.additionalCount = htons(data.additionalCount);
      return;
    }
    

  }  // namespace Dns

}  // namespace Dwm
