//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.5/classes/include/DwmDnsRRDataMINFO.hh 10129 $
// @(#) $Id: DwmDnsRRDataMINFO.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataMINFO.hh
//!  \brief Dwm::Dns::RRDataMINFO class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATAMINFO_HH_
#define _DWMDNSRRDATAMINFO_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates MINFO resource record data.  See RFC1035 for details.
    //------------------------------------------------------------------------
    class RRDataMINFO
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 14;
      
      //----------------------------------------------------------------------
      //!  Construct MINFO resource record data with empty RMAILBX and
      //!  empty EMAILBX.
      //----------------------------------------------------------------------
      RRDataMINFO();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c rmailbx and @c emailbx.
      //----------------------------------------------------------------------
      RRDataMINFO(const std::string & rmailbx, const std::string & emailbx);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataMINFO & minfo) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained RMAILBX.
      //----------------------------------------------------------------------
      const std::string & Rmailbx() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained RMAILBX.
      //----------------------------------------------------------------------
      const std::string & Rmailbx(const std::string & rmailbx);
      
      //----------------------------------------------------------------------
      //!  Returns the contained EMAILBX.
      //----------------------------------------------------------------------
      const std::string & Emailbx() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained EMAILBX.
      //----------------------------------------------------------------------
      const std::string & Emailbx(const std::string & emailbx);
      
      //----------------------------------------------------------------------
      //!  Encodes the MINFO resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded MINFO resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded MINFO resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the MINFO resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  MINFO resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded MINFO resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);
      
      //----------------------------------------------------------------------
      //!  Print the MINFO resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataMINFO & minfo);
      
    private:
      std::string  _rmailbx;
      std::string  _emailbx;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATAMINFO_HH_
