//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/tests/TestDnsResourceRecord.cc 10029 $
// @(#) $Id: TestDnsResourceRecord.cc 10029 2018-01-17 06:44:47Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestDnsResourceRecord.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
}

#include <sstream>

#include "DwmSvnTag.hh"
#include "DwmUnitAssert.hh"
#include "DwmDnsResourceRecord.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/tests/TestDnsResourceRecord.cc 10029 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestEncodeDecode()
{
  const vector<pair<string,Dns::ResourceRecord::TypeEnum>>  addrs = {
    { "::1",                   Dns::ResourceRecord::k_typeAAAA },
    { "mail.mcplex.net",       Dns::ResourceRecord::k_typeMB },
    { "group.mcplex.net",      Dns::ResourceRecord::k_typeMG },
    { "127.0.0.1",             Dns::ResourceRecord::k_typeA },
    { "ria.rfdm.com",          Dns::ResourceRecord::k_typeNS },
    { "fd60:3019:f4a:6aaf::1", Dns::ResourceRecord::k_typeAAAA },
    { "dwm.mcplex.net",        Dns::ResourceRecord::k_typeMR },
    { "ptr.mcplex.net",        Dns::ResourceRecord::k_typePTR },
    { "ns1.google.com",        Dns::ResourceRecord::k_typeNS },
    { "192.168.168.1",         Dns::ResourceRecord::k_typeA },
    { "www.rfdm.com",          Dns::ResourceRecord::k_typeCNAME }
  };
  uint8_t   pkt[4096] = { 0 };
  uint8_t  *ptr = pkt;
  
  vector<Dns::ResourceRecord>  rrs;
  for (auto addr : addrs) {
    Dns::ResourceRecord  rr;
    in6_addr             in6Addr;
    in_addr              inAddr;
    switch (addr.second) {
      case Dns::ResourceRecord::k_typeAAAA:
        inet_pton(AF_INET6, addr.first.c_str(), &in6Addr);
        rr.Name("rfdm.com");
        rr.Data(Dns::RRDataAAAA(in6Addr));
        rrs.push_back(rr);
        break;
      case Dns::ResourceRecord::k_typeA:
        inet_pton(AF_INET, addr.first.c_str(), &inAddr);
        rr.Name("rfdm.com");
        rr.Data(Dns::RRDataA(inAddr));
        rrs.push_back(rr);
        break;
      case Dns::ResourceRecord::k_typeNS:
        rr.Name(addr.first.substr(addr.first.find_first_of('.') + 1));
        rr.Data(Dns::RRDataNS(addr.first));
        rrs.push_back(rr);
        break;
      case Dns::ResourceRecord::k_typeCNAME:
        rr.Name(addr.first.substr(addr.first.find_first_of('.') + 1));
        rr.Data(Dns::RRDataCNAME(addr.first));
        rrs.push_back(rr);
        break;
      case Dns::ResourceRecord::k_typeMB:
        rr.Name(addr.first);
        rr.Data(Dns::RRDataMB(addr.first));
        rrs.push_back(rr);
        break;
      case Dns::ResourceRecord::k_typeMG:
        rr.Name(addr.first);
        rr.Data(Dns::RRDataMG(addr.first));
        rrs.push_back(rr);
        break;
      case Dns::ResourceRecord::k_typeMR:
        rr.Name(addr.first);
        rr.Data(Dns::RRDataMR(addr.first));
        rrs.push_back(rr);
        break;
      case Dns::ResourceRecord::k_typePTR:
        rr.Name(addr.first);
        rr.Data(Dns::RRDataPTR(addr.first));
        rrs.push_back(rr);
        break;
      default:
        break;
    }
  }

  ostringstream  rross;
  Dns::LabelPositions  lps;
  for (auto & rr : rrs) {
    ptr = rr.Encode(pkt, ptr, 4096, lps);
    rross << rr << '\n';
  }

  Dns::RRDataSOA  soaData("mcplex.net", "dwm.mcplex.net", 2017011000, 3600,
                          1800, 7200, 3600);
  Dns::ResourceRecord  soarr;
  soarr.Data(soaData);
  ptr = soarr.Encode(pkt, ptr, 4096, lps);
  rross << soarr << '\n';

  Dns::RRDataHINFO hinfoData("amd64", "FreeBSD_11.1-STABLE");
  Dns::ResourceRecord  hinforr;
  hinforr.Name("kiva.rfdm.com");
  hinforr.Data(hinfoData);
  ptr = hinforr.Encode(pkt, ptr, 4096, lps);
  rross << hinforr << '\n';

  Dns::RRDataMINFO minfoData("folks.mcplex.net", "errors.mcplex.net");
  Dns::ResourceRecord  minforr;
  minforr.Name("folks.mcplex.net");
  minforr.Data(minfoData);
  ptr = minforr.Encode(pkt, ptr, 4096, lps);
  rross << minforr << '\n';
  
  Dns::RRDataMX  mxData(10, "mail.mcplex.net");
  Dns::ResourceRecord  mxrr;
  mxrr.Name("mcplex.net");
  mxrr.Data(mxData);
  ptr = mxrr.Encode(pkt, ptr, 4096, lps);
  rross << mxrr << '\n';

  Dns::RRDataTXT  txtData(vector<string>{"random text","in a text record",
                                         "for testing"});
  Dns::ResourceRecord  txtrr;
  txtrr.Name("txt.mcplex.net");
  txtrr.Data(txtData);
  ptr = txtrr.Encode(pkt, ptr, 4096, lps);
  rross << txtrr << '\n';

  Dns::RRDataRP  rpData("mbox.mcplex.net", "txt.mbox.mcplex.net");
  Dns::ResourceRecord  rprr;
  rprr.Name("mbox.mcplex.net");
  rprr.Data(rpData);
  ptr = rprr.Encode(pkt, ptr, 4096, lps);
  rross << rprr << '\n';

  Dns::RRDataKEY  keyData(0, 1, 1, 0, "not a real key, but for testing");
  Dns::ResourceRecord  keyrr;
  keyrr.Name("key.rfdm.com");
  keyrr.Data(keyData);
  ptr = keyrr.Encode(pkt, ptr, 4096, lps);
  rross << keyrr << '\n';

  Dns::RRDataLOC  locData(0, 1, 1, 1, 10, 20, 100100 * 100);
  Dns::ResourceRecord  locrr;
  locrr.Name("loc.mcplex.net");
  locrr.Data(locData);
  ptr = locrr.Encode(pkt, ptr, 4096, lps);
  rross << locrr << '\n';

  Dns::RRDataSRV  srvData(10, 20, 443, "www.rfdm.com");
  Dns::ResourceRecord  srvrr;
  srvrr.Name("_smtp._tcp.rfdm.com");
  srvrr.Data(srvData);
  ptr = srvrr.Encode(pkt, ptr, 4096, lps);
  rross << srvrr << '\n';

  Dns::RRDataCERT  certData(Dns::RRDataCERT::k_typePGP, 0,
                            Dns::RRDataCERT::k_algorithmDSA,
                            "Garbage certificate data, just for testing");
  Dns::ResourceRecord  certrr;
  certrr.Data(certData);
  ptr = certrr.Encode(pkt, ptr, 4096, lps);
  rross << certrr << '\n';

  Dns::RRDataDS  dsData(0, Dns::RRDataDS::k_algorithmRSASHA1,
                        Dns::RRDataDS::k_digestTypeSHA1,
                        "Garbage data, just for testing");
  Dns::ResourceRecord  dsrr;
  dsrr.Data(dsData);
  ptr = dsrr.Encode(pkt, ptr, 4096, lps);
  rross << dsrr << '\n';

  Dns::RRDataSSHFP  sshfpData(Dns::RRDataSSHFP::k_algorithmRSA,
                              Dns::RRDataSSHFP::k_fpTypeSHA1,
                              "Garbage SSHFP fingerprint data, just for testing");
  Dns::ResourceRecord  sshfprr;
  sshfprr.Data(sshfpData);
  ptr = sshfprr.Encode(pkt, ptr, 4096, lps);
  rross << sshfprr << '\n';

  Dns::NSECBitmap  nsecBitmap0(0, std::vector<uint8_t>{15, 7, 16, 10});
  Dns::NSECBitmap  nsecBitmap1(1, std::vector<uint8_t>{8, 3, 17});
  vector<Dns::NSECBitmap>  bitmaps = { nsecBitmap0, nsecBitmap1 };
  Dns::RRDataNSEC  nsecData("whatever.rfdm.com", bitmaps);
  Dns::ResourceRecord  nsecrr;
  nsecrr.Data(nsecData);
  ptr = nsecrr.Encode(pkt, ptr, 4096, lps);
  rross << nsecrr << '\n';

  Dns::RRDataDNSKEY  dnskeyData(0, 3, Dns::RRDataDNSKEY::k_algorithmRSASHA1,
                                "garbage RSA public key for testing");
  Dns::ResourceRecord  dnskeyrr;
  dnskeyrr.Data(dnskeyData);
  ptr = dnskeyrr.Encode(pkt, ptr, 4096, lps);
  rross << dnskeyrr << '\n';

  Dns::RRDataRRSIG  rrsigData(1, 2, 3, 10, 11, 5, 42, "dwm.rfdm.com",
                              "garbage RRSIG for testing");
  Dns::ResourceRecord  rrsigrr;
  rrsigrr.Data(rrsigData);
  ptr = rrsigrr.Encode(pkt, ptr, 4096, lps);
  rross << rrsigrr << '\n';

  Dns::RRDataDHCID  dhcidData(1, 1, "garbage DHCID for testing");
  Dns::ResourceRecord  dhcidrr;
  dhcidrr.Data(dhcidData);
  ptr = dhcidrr.Encode(pkt, ptr, 4096, lps);
  rross << dhcidrr << '\n';

  Dns::RRDataNSEC3  nsec3Data(1, 1, 8, "TeStSaLt", "TeStHaShEdOwNeRnAmE",
                              bitmaps);
  Dns::ResourceRecord  nsec3rr;
  nsec3rr.Data(nsec3Data);
  ptr = nsec3rr.Encode(pkt, ptr, 4096, lps);
  rross << nsec3rr << '\n';

  Dns::RRDataNSEC3PARAM  nsec3parData(1, 1, 8, "TeStSaLt");
  Dns::ResourceRecord  nsec3parrr;
  nsec3parrr.Data(nsec3parData);
  ptr = nsec3parrr.Encode(pkt, ptr, 4096, lps);
  rross << nsec3parrr << '\n';

  Dns::RRDataTLSA  tlsaData(1, 0, 0,
                            "bogus test certificate data for ca.mcplex.net");
  Dns::ResourceRecord  tlsarr;
  tlsarr.Data(tlsaData);
  ptr = tlsarr.Encode(pkt, ptr, 4096, lps);
  rross << tlsarr << '\n';

  Dns::RRDataSMIMEA  smimeaData(1, 0, 0, "bogus SMIMEA test certificate data");
  Dns::ResourceRecord  smimearr;
  smimearr.Data(smimeaData);
  ptr = smimearr.Encode(pkt, ptr, 4096, lps);
  rross << smimearr << '\n';

  Dns::RRDataOPENPGPKEY  openpgpData("junkTestPGPPublicKeyString");
  Dns::ResourceRecord  openpgpkeyrr;
  openpgpkeyrr.Data(openpgpData);
  ptr = openpgpkeyrr.Encode(pkt, ptr, 4096, lps);
  rross << openpgpkeyrr << '\n';

  Dns::RRDataURI  uriData(1, 10, "http://www.mcplex.net/target");
  Dns::ResourceRecord  urirr;
  urirr.Data(uriData);
  ptr = urirr.Encode(pkt, ptr, 4096, lps);
  rross << urirr << '\n';
  
  Dns::RRDataCAA  caaData(1, "issue", "ca.mcplex.net");
  Dns::ResourceRecord  caarr;
  caarr.Data(caaData);
  ptr = caarr.Encode(pkt, ptr, 4096, lps);
  rross << caarr << '\n';

  Dns::RRDataUnknown  unknownData("unknown data");
  Dns::ResourceRecord  unknownrr;
  unknownrr.Data(unknownData, Dns::ResourceRecord::k_classIN, 424);
  ptr = unknownrr.Encode(pkt, ptr, 4096, lps);
  rross << unknownrr << '\n';
  
  //------  Decoding ------
  ostringstream  dross;
  const uint8_t  *rptr = pkt;
  for (auto & rr : rrs) {
    Dns::ResourceRecord  rrd;
    rptr = rrd.Decode(pkt, rptr, 4096);
    dross << rrd << '\n';
    UnitAssert(rrd == rr);
  }
  Dns::ResourceRecord  rrr;
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == soarr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == hinforr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == minforr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == mxrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == txtrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == rprr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == keyrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == locrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == srvrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == certrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == dsrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == sshfprr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == nsecrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == dnskeyrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == rrsigrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == dhcidrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == nsec3rr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == nsec3parrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == tlsarr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == smimearr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == openpgpkeyrr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == urirr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == caarr);
  rptr = rrr.Decode(pkt, rptr, 4096);
  dross << rrr << '\n';
  UnitAssert(rrr == unknownrr);
  
  UnitAssert(rptr == ptr);
  UnitAssert(dross.str() == rross.str());
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  TestEncodeDecode();
  
  if (Assertions::Total().Failed()) {
    Assertions::Print(cerr, true);
    return 1;
  }
  else {
    cout << Assertions::Total() << " passed" << endl;
    return 0;
  }
  
}
