//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/src/DwmDnsRRDataURI.cc 10133 $
// @(#) $Id: DwmDnsRRDataURI.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataURI.cc
//!  \brief Dwm::Dns::RRDataURI class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>

#include "DwmSvnTag.hh"
#include "DwmDnsRRDataURI.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/src/DwmDnsRRDataURI.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataURI::RRDataURI()
        : _priority(0), _weight(0), _target()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataURI::RRDataURI(uint16_t priority, uint16_t weight,
                         const string & target)
        : _priority(priority), _weight(weight), _target(target)
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataURI::operator == (const RRDataURI & uri) const
    {
      return ((_priority == uri._priority)
              && (_weight == uri._weight)
              && (_target == uri._target));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataURI::Priority() const
    {
      return _priority;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataURI::Priority(uint16_t priority)
    {
      _priority = priority;
      return _priority;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataURI::Weight() const
    {
      return _weight;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataURI::Weight(uint16_t weight)
    {
      _weight = weight;
      return _weight;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataURI::Target() const
    {
      return _target;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataURI::Target(const string & target)
    {
      _target = target;
      return _target;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataURI::Encode(uint8_t *pkt, uint8_t *ptr,
                               uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 6) <= (pkt + pktlen)) {
        uint16_t  val = htons(_priority);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        val = htons(_weight);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        if (! _target.empty()) {
          if ((ptr + _target.empty() <= (pkt + pktlen))) {
            memcpy(ptr, _target.c_str(), _target.size());
            ptr += _target.size();
          }
          else {
            throw out_of_range("Dwm::Dns::RRDataURI will not fit in packet");
          }
        }
      }
      else {
        throw out_of_range("Dwm::Dns::RRDataURI will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataURI::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                     uint16_t pktlen, uint16_t rdlen)
    {
      _priority = 0;
      _weight = 0;
      _target.clear();

      const uint8_t  *origptr = ptr;
      if ((ptr + 6) <= (pkt + pktlen)) {
        memcpy(&_priority, ptr, sizeof(_priority));
        _priority = ntohs(_priority);
        ptr += sizeof(_priority);
        memcpy(&_weight, ptr, sizeof(_weight));
        _weight = ntohs(_weight);
        ptr += sizeof(_weight);
        if ((ptr <= (origptr + rdlen)) && (ptr <= (ptr + pktlen))) {
          int  tlen = rdlen - (ptr - origptr);
          _target.assign((const char *)ptr, tlen);
          ptr += tlen;
        }
        else {
          throw out_of_range("packet too short to contain"
                             " Dwm::Dns::RRDataURI");
        }
      }
      else {
        throw out_of_range("packet too short to contain Dwm::Dns::RRDataURI");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataURI & uri)
    {
      if (os) {
        os << uri._priority << ' ' << uri._weight << ' ' << uri._target;
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
