//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/src/DwmDnsRRDataNSEC3PARAM.cc 10133 $
// @(#) $Id: DwmDnsRRDataNSEC3PARAM.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataNSEC3PARAM.cc
//!  \brief Dwm::Dns::RRDataNSEC3PARAM class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <iomanip>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataNSEC3PARAM.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/src/DwmDnsRRDataNSEC3PARAM.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataNSEC3PARAM::RRDataNSEC3PARAM()
        : _algorithm(0), _flags(0), _iterations(0), _salt()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataNSEC3PARAM::RRDataNSEC3PARAM(uint8_t algorithm, uint8_t flags,
                                       uint16_t iterations,
                                       const std::string & salt)
        : _algorithm(algorithm), _flags(flags), _iterations(iterations),
          _salt(salt)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool
    RRDataNSEC3PARAM::operator == (const RRDataNSEC3PARAM & nsec3param) const
    {
      return ((_algorithm == nsec3param._algorithm)
              && (_flags == nsec3param._flags)
              && (_iterations == nsec3param._iterations)
              && (_salt == nsec3param._salt));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3PARAM::Algorithm() const
    {
      return _algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3PARAM::Algorithm(uint8_t algorithm)
    {
      _algorithm = algorithm;
      return _algorithm;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3PARAM::Flags() const
    {
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataNSEC3PARAM::Flags(uint8_t flags)
    {
      _flags = flags;
      return _flags;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataNSEC3PARAM::Iterations() const
    {
      return _iterations;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataNSEC3PARAM::Iterations(uint16_t iterations)
    {
      _iterations = iterations;
      return _iterations;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RRDataNSEC3PARAM::Salt() const
    {
      return _salt;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RRDataNSEC3PARAM::Salt(const std::string & salt)
    {
      _salt = salt;
      return _salt;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataNSEC3PARAM::Encode(uint8_t *pkt, uint8_t *ptr,
                                      uint16_t pktlen,
                                      LabelPositions & lps) const
    {
      if ((ptr + 5) <= (pkt + pktlen)) {
        *ptr++ = _algorithm;
        *ptr++ = _flags;
        uint16_t  val = htons(_iterations);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        uint8_t  len = (uint8_t)(_salt.size() & 0xFF);
        *ptr++ = len;
        if (len > 0) {
          if ((ptr + len) <= (pkt + pktlen)) {
            memcpy(ptr, _salt.c_str(), len);
            ptr += len;
          }
          else {
            throw out_of_range("Dwm::Dns::RRDataNSEC3PARAM will not fit"
                               " in packet");
          }
        }
      }
      else {
        throw out_of_range("Dwm::Dns::RRDataNSEC3PARAM will not fit"
                           " in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataNSEC3PARAM::Decode(const uint8_t *pkt,
                                            const uint8_t *ptr,
                                            uint16_t pktlen, uint16_t rdlen)
    {
      _algorithm = 0;
      _flags = 0;
      _iterations = 0;
      _salt.clear();
      
      if ((ptr + 5) <= (pkt + pktlen)) {
        _algorithm = *ptr++;
        _flags = *ptr++;
        memcpy(&_iterations, ptr, sizeof(_iterations));
        _iterations = ntohs(_iterations);
        ptr += sizeof(_iterations);
        uint8_t  len = *ptr++;
        if (len > 0) {
          if ((ptr + len) <= (pkt + pktlen)) {
            _salt.assign((const char *)ptr, len);
            ptr += len;
          }
          else {
            throw out_of_range("packet too short to contain"
                               " Dwm::Dns::RRDataNSEC3PARAM");
          }
        }
      }
      else {
        throw out_of_range("packet too short to contain"
                           " Dwm::Dns::RRDataNSEC3PARAM");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataNSEC3PARAM & param)
    {
      if (os) {
        os << (uint16_t)param._algorithm << ' '
           << "0x" << hex << (uint16_t)param._flags << dec << ' '
           << param._iterations << Base64::Encode(param._salt);
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
