//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/include/DwmDnsRRDataNSEC3PARAM.hh 10132 $
// @(#) $Id: DwmDnsRRDataNSEC3PARAM.hh 10132 2018-01-27 17:25:04Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataNSEC3PARAM.hh
//!  \brief Dwm::Dns::RRDataNSEC3PARAM class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATANSEC3PARAM_HH_
#define _DWMDNSRRDATANSEC3PARAM_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates NSEC3PARAM resource record data.  See RFC5155 for more
    //!  details.
    //------------------------------------------------------------------------
    class RRDataNSEC3PARAM
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 51;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty salt.
      //----------------------------------------------------------------------
      RRDataNSEC3PARAM();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c algorithm, @c flags, @c iterations
      //!  and @c salt.
      //----------------------------------------------------------------------
      RRDataNSEC3PARAM(uint8_t algorithm, uint8_t flags, uint16_t iterations,
                       const std::string & salt);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataNSEC3PARAM & nsec3param) const;
      
      //----------------------------------------------------------------------
      //!  Returns the algorithm in the NSEC3PARAM resource record data.
      //----------------------------------------------------------------------
      uint8_t Algorithm() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the algorithm in the NSEC3PARAM resource record
      //!  data.
      //----------------------------------------------------------------------
      uint8_t Algorithm(uint8_t algorithm);
      
      //----------------------------------------------------------------------
      //!  Returns the flags in the resource record data.
      //----------------------------------------------------------------------
      uint8_t Flags() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the flags in the resource record data.
      //----------------------------------------------------------------------
      uint8_t Flags(uint8_t flags);
      
      //----------------------------------------------------------------------
      //!  Returns the iterations in the resource record data.
      //----------------------------------------------------------------------
      uint16_t Iterations() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the iterations in the resource record data.
      //----------------------------------------------------------------------
      uint16_t Iterations(uint16_t iterations);
      
      //----------------------------------------------------------------------
      //!  Returns the salt in the resource record data.
      //----------------------------------------------------------------------
      const std::string & Salt() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the salt in the resource record data.
      //----------------------------------------------------------------------
      const std::string & Salt(const std::string & salt);
      
      //----------------------------------------------------------------------
      //!  Encodes the NSEC3PARAM resource record data into buffer @ pkt of
      //!  length @c pktlen, starting at @c ptr.  Returns the address
      //!  immediately following the encoded NSEC3PARAM resource record data
      //!  in @c pkt on success.  Throws std::out_of_range if @c pkt is too
      //!  short to contain the encoded NSEC3PARAM resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the NSEC3PARAM resource record data from buffer @c pkt of
      //!  length @c pktlen, starting at @c ptr.  Returns the address
      //!  immediately following the encoded NSEC3PARAM resource record data
      //!  in @c pkt on success.  Throws std::out_of_range if @c pkt was too
      //!  short to contain an encoded NSEC3PARAM resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the NSEC3PARAM resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataNSEC3PARAM & param);
      
    private:
      uint8_t       _algorithm;
      uint8_t       _flags;
      uint16_t      _iterations;
      std::string   _salt;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATANSEC3PARAM_HH_
