//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/include/DwmDnsRRDataCERT.hh 10129 $
// @(#) $Id: DwmDnsRRDataCERT.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataCERT.hh
//!  \brief Dwm::Dns::RRDataCERT class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATACERT_HH_
#define _DWMDNSRRDATACERT_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates CERT resource record data.  See RFC 4398 for details.
    //   xxx - TODO: breakdown of cerrtificate data, which varies by type.
    //------------------------------------------------------------------------
    class RRDataCERT
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 37;
      
      static const uint16_t  k_typePKIX         = 1;
      static const uint16_t  k_typeSPKI         = 2;
      static const uint16_t  k_typePGP          = 3;
      static const uint16_t  k_typeIPKIX        = 4;
      static const uint16_t  k_typeISPKI        = 5;
      static const uint16_t  k_typeIPGP         = 6;
      static const uint16_t  k_typeACPKIX       = 7;
      static const uint16_t  k_typeIACPKIX      = 8;
      static const uint16_t  k_typeURI          = 253;
      static const uint16_t  k_typeOID          = 254;
      
      static const uint8_t   k_algorithmReserved  = 0;
      static const uint8_t   k_algorithmRSAMD5    = 1;
      static const uint8_t   k_algorithmDH        = 2;
      static const uint8_t   k_algorithmDSA       = 3;
      static const uint8_t   k_algorithmECC       = 4;
      
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      RRDataCERT();
      
      //----------------------------------------------------------------------
      //!  Construct a CERT with @c type, @c keyTag, @c algorithm and
      //!  @c certificate.
      //----------------------------------------------------------------------
      RRDataCERT(uint16_t type, uint16_t keyTag, uint8_t algorithm,
                const std::string & certificate);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataCERT & cert) const;
      
      //----------------------------------------------------------------------
      //!  Returns the type in the CERT resource record data.
      //----------------------------------------------------------------------
      uint16_t Type() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the type in the CERT resource record data.
      //----------------------------------------------------------------------
      uint16_t Type(uint16_t type);
      
      //----------------------------------------------------------------------
      //!  Returns the key tag in the CERT resource record data.
      //----------------------------------------------------------------------
      uint16_t KeyTag() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the key tag in the CERT resource record data.
      //----------------------------------------------------------------------
      uint16_t KeyTag(uint16_t keyTag);
      
      //----------------------------------------------------------------------
      //!  Returns the algorithm in the CERT resource record data.
      //----------------------------------------------------------------------
      uint8_t Algorithm() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the algorithm in the CERT resource record data.
      //----------------------------------------------------------------------
      uint8_t Algorithm(uint8_t algorithm);
      
      //----------------------------------------------------------------------
      //!  Returns the certificate in the CERT resource record data.
      //----------------------------------------------------------------------
      const std::string & Certificate() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the certificate in the CERT resource record data.
      //----------------------------------------------------------------------
      const std::string & Certificate(const std::string & certificate);
      
      //----------------------------------------------------------------------
      //!  Encodes the CERT resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded CERT resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded CERT resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the CERT resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  CERT resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded CERT resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the CERT resource record data to an ostream in human readable
      //!  form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataCERT & cert);
      
    private:
      uint16_t     _type;
      uint16_t     _keyTag;
      uint8_t      _algorithm;
      std::string  _certificate;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATACERT_HH_
