//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.2/classes/include/DwmDnsNSECBitmap.hh 10089 $
// @(#) $Id: DwmDnsNSECBitmap.hh 10089 2018-01-26 04:51:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsNSECBitmap.hh
//!  \brief Dwm::Dns::NSECBitmap class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSNSECBITMAP_HH_
#define _DWMDNSNSECBITMAP_HH_

#include <cstdint>
#include <iostream>
#include <vector>

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  Encapsulates an NSEC bitmap.  See section 4.1.2 of RFC4034.
    //------------------------------------------------------------------------
    class NSECBitmap
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct an empty bitmap.
      //----------------------------------------------------------------------
      NSECBitmap();
      
      //----------------------------------------------------------------------
      //!  Construct a bitmap with window number @c window and bitmap @c bits.
      //----------------------------------------------------------------------
      NSECBitmap(uint8_t window, const std::vector<uint8_t> & bits);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const NSECBitmap & bitmap) const;
      
      //----------------------------------------------------------------------
      //!  Returns the window number of the bitmap.
      //----------------------------------------------------------------------
      uint8_t Window() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the window number of the bitmap.
      //----------------------------------------------------------------------
      uint8_t Window(uint8_t window);
      
      //----------------------------------------------------------------------
      //!  Returns the bits in the bitmap.
      //----------------------------------------------------------------------
      const std::vector<uint8_t> & Bits() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the bits in the bitmap.
      //----------------------------------------------------------------------
      const std::vector<uint8_t> & Bits(const std::vector<uint8_t> & bits);
      
      //----------------------------------------------------------------------
      //!  Encodes the bitmap into buffer @ pkt of length @c pktlen, starting
      //!  at @c ptr.  Returns the address immediately following the encoded
      //!  bitmap in @c pkt on success.  Throws std::out_of_range if @c pkt
      //!  is too short to contain the encoded bitmap.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the bitmap from buffer @c pkt of length @c pktlen, starting
      //!  at @c ptr.  Returns the address immediately following the encoded
      //!  bitmap in @c pkt on success.  Throws std::out_of_range if @c pkt
      //!  was too short to contain an encoded bitmap.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Prints the bitmap to an ostream in human-readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const NSECBitmap & bitmap);
      
    private:
      uint8_t               _window;
      std::vector<uint8_t>  _bits;
    };

  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSNSECBITMAP_HH_
