//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmDnsResolver.cc 10719 2020-05-06 19:43:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsResolver.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmDnsResolver.hh"
#include "DwmDnsResolvConf.hh"
#include "DwmDnsUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.1/classes/src/DwmDnsResolver.cc 10719 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Resolver::Resolver(const string & filename, const string & order)
        : _order(order)
    {
      _msgid.store(1);
      if (_order.empty()) {
        _order = "files,dns";
      }
      ResolvConf  resconf(filename);
      for (auto & ns : resconf.Nameservers()) {
        _nameservers.push_back(std::move(ns));
      }
      for (auto & sle : resconf.SearchList()) {
        _searchList.push_back(sle);
      }
      _domain = resconf.Domain();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Resolver::GetHostByName(const string & name,
                                 vector<in6_addr> & in6Addrs,
                                 vector<in_addr> & inAddrs)
    {
      in6Addrs.clear();
      inAddrs.clear();
      std::vector<ResourceRecord>  rrs;
      if (Get(name, ResourceRecord::k_typeAAAA, rrs)) {
        for (auto & rr : rrs) {
          in6Addrs.push_back(rr.Data<RRDataAAAA>()->In6Addr());
        }
      }
      if (Get(name, ResourceRecord::k_typeA, rrs)) {
        for (auto rr : rrs) {
          inAddrs.push_back(rr.Data<RRDataA>()->InAddr());
        }
      }
      return ((! in6Addrs.empty()) || (! inAddrs.empty()));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Resolver::GetHostByAddr(const in6_addr & addr,
                                 vector<string> & names)
    {
      names.clear();
      vector<ResourceRecord>  rrs;
      string                  arpa;
      ToArpa(addr, arpa);
      if (Get(arpa, ResourceRecord::k_typePTR, rrs)) {
        for (auto & rr : rrs) {
          names.push_back(rr.Data<RRDataPTR>()->Ptr());
        }
      }
      return (! names.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Resolver::GetHostByAddr(const in_addr & addr,
                                 vector<string> & names)
    {
      names.clear();
      vector<ResourceRecord>  rrs;
      string                  arpa;
      ToArpa(addr, arpa);
      if (Get(arpa, ResourceRecord::k_typePTR, rrs)) {
        for (auto & rr : rrs) {
          names.push_back(rr.Data<RRDataPTR>()->Ptr());
        }
      }
      return (! names.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Resolver::Get(const string & name, uint16_t rrtype,
                       vector<ResourceRecord> & rrs, bool de,
                       bool dd, bool fb)
    {
      using RR = ResourceRecord;
      rrs.clear();
      switch (rrtype) {
        case RR::k_typeA:
          Get<RRDataA>(name, rrs, de, dd, fb);                       break;
        case RR::k_typeNS:
          Get<RRDataNS>(name, rrs, de, dd, fb);                      break;
        case RR::k_typeCNAME:
          Get<RRDataCNAME>(name, rrs, de, dd, fb);                   break;
        case RR::k_typeAAAA:
          Get<RRDataAAAA>(name, rrs, de, dd, fb);                    break;
        case RR::k_typeSOA:
          Get<RRDataSOA>(name, rrs, de, dd, fb);                     break;
        case RR::k_typeMB:
          Get<RRDataMB>(name, rrs, de, dd, fb);                      break;
        case RR::k_typeMG:
          Get<RRDataMG>(name, rrs, de, dd, fb);                      break;
        case RR::k_typeMR:
          Get<RRDataMR>(name, rrs, de, dd, fb);                      break;
        case RR::k_typePTR:
          Get<RRDataPTR>(name, rrs, de, dd, fb);                     break;
        case RR::k_typeHINFO:
          Get<RRDataHINFO>(name, rrs, de, dd, fb);                   break;
        case RR::k_typeMINFO:
          Get<RRDataMINFO>(name, rrs, de, dd, fb);                   break;
        case RR::k_typeMX:
          Get<RRDataMX>(name, rrs, de, dd, fb);                      break;
        case RR::k_typeTXT:
          Get<RRDataTXT>(name, rrs, de, dd, fb);                     break;
        case RR::k_typeRP:
          Get<RRDataRP>(name, rrs, de, dd, fb);                      break;
        case RR::k_typeKEY:
          Get<RRDataKEY>(name, rrs, de, dd, fb);                     break;
        case RR::k_typeLOC:
          Get<RRDataLOC>(name, rrs, de, dd, fb);                     break;
        case RR::k_typeSRV:
          Get<RRDataSRV>(name, rrs, de, dd, fb);                     break;
        case RR::k_typeCERT:
          Get<RRDataCERT>(name, rrs, de, dd, fb);                    break;
        case RR::k_typeDS:
          Get<RRDataDS>(name, rrs, de, dd, fb);                      break;
        case RR::k_typeSSHFP:
          Get<RRDataSSHFP>(name, rrs, de, dd, fb);                   break;
        case RR::k_typeRRSIG:
          Get<RRDataRRSIG>(name, rrs, de, dd, fb);                   break;
        case RR::k_typeNSEC:
          Get<RRDataNSEC>(name, rrs, de, dd, fb);                    break;
        case RR::k_typeDNSKEY:
          Get<RRDataDNSKEY>(name, rrs, de, dd, fb);                  break;
        case RR::k_typeDHCID:
          Get<RRDataDHCID>(name, rrs, de, dd, fb);                   break;
        case RR::k_typeNSEC3:
          Get<RRDataNSEC3>(name, rrs, de, dd, fb);                   break;
        case RR::k_typeNSEC3PARAM:
          Get<RRDataNSEC3PARAM>(name, rrs, de, dd, fb);              break;
        case RR::k_typeTLSA:
          Get<RRDataTLSA>(name, rrs, de, dd, fb);                    break;
        case RR::k_typeSMIMEA:
          Get<RRDataSMIMEA>(name, rrs, de, dd, fb);                  break;
        case RR::k_typeOPENPGPKEY:
          Get<RRDataOPENPGPKEY>(name, rrs, de, dd, fb);              break;
        case RR::k_typeURI:
          Get<RRDataURI>(name, rrs, de, dd, fb);                     break;
        case RR::k_typeCAA:
          Get<RRDataCAA>(name, rrs, de, dd, fb);                     break;
        default:
          break;
      }
      return (! rrs.empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Resolver::GetNamesToTry(const string & name, uint16_t rrtype,
                                 vector<string> & names) const
    {
      names.clear();
      if (rrtype != ResourceRecord::k_typePTR) {
        if ((name.find_first_of('.') != string::npos)
            || (name == "localhost")) {
          names.push_back(name);
        }
        for (auto & sl : _searchList) {
          names.push_back(name + "." + sl);
        }
        if (_searchList.empty()) {
          if (! _domain.empty()) {
            names.push_back(name + "." + _domain);
          }
        }
      }
      else {
        names.push_back(name);
      }
    }
    
  }  // namespace Dns

}  // namespace Dwm
