//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.1/classes/src/DwmDnsRRDataSMIMEA.cc 10075 $
// @(#) $Id: DwmDnsRRDataSMIMEA.cc 10075 2018-01-25 02:59:37Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataSMIMEA.cc
//!  \brief Dwm::Dns::RRDataSMIMEA class implementation
//---------------------------------------------------------------------------

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataSMIMEA.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.1/classes/src/DwmDnsRRDataSMIMEA.cc 10075 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataSMIMEA::RRDataSMIMEA()
        : _certificateUsage(0), _selector(0), _matchingType(0),
          _certificateAssociation()
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataSMIMEA::RRDataSMIMEA(uint8_t certificateUsage, uint8_t selector,
                           uint8_t matchingType, 
                           const string & certificateAssociation)
        : _certificateUsage(certificateUsage), _selector(selector),
          _matchingType(matchingType),
          _certificateAssociation(certificateAssociation)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataSMIMEA::operator == (const RRDataSMIMEA & smimea) const
    {
      return ((_certificateUsage == smimea._certificateUsage)
              && (_selector == smimea._selector)
              && (_matchingType == smimea._matchingType)
              && (_certificateAssociation == smimea._certificateAssociation));
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataSMIMEA::CertificateUsage() const
    {
      return _certificateUsage;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataSMIMEA::CertificateUsage(uint8_t certificateUsage)
    {
      _certificateUsage = certificateUsage;
      return _certificateUsage;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataSMIMEA::Selector() const
    {
      return _selector;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataSMIMEA::Selector(uint8_t selector)
    {
      _selector = selector;
      return _selector;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataSMIMEA::MatchingType() const
    {
      return _matchingType;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataSMIMEA::MatchingType(uint8_t matchingType)
    {
      _matchingType = matchingType;
      return _matchingType;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataSMIMEA::CertificateAssociation() const
    {
      return _certificateAssociation;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string &
    RRDataSMIMEA::CertificateAssociation(const string & certificateAssociation)
    {
      _certificateAssociation = certificateAssociation;
      return _certificateAssociation;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataSMIMEA::Encode(uint8_t *pkt, uint8_t *ptr,
                                  uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 3) <= (pkt + pktlen)) {
        *ptr++ = _certificateUsage;
        *ptr++ = _selector;
        *ptr++ = _matchingType;
        if ((ptr + _certificateAssociation.size()) <= (pkt + pktlen)) {
          if (! _certificateAssociation.empty()) {
            memcpy(ptr, _certificateAssociation.c_str(),
                   _certificateAssociation.size());
            ptr += _certificateAssociation.size();
          }
        }
        else {
          throw out_of_range("Dwm::Dns::RRDataSMIMEA will not fit in packet");
        }
      }
      else {
        throw	out_of_range("Dwm::Dns::RRDataSMIMEA will not fit in packet");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataSMIMEA::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                      uint16_t pktlen, uint16_t rdlen)
    {
      _certificateUsage = 0;
      _selector = 0;
      _matchingType = 0;

      const uint8_t  *origptr = ptr;
      if ((ptr + 3) <= (pkt + pktlen)) {
        _certificateUsage = *ptr++;
        _selector = *ptr++;
        _matchingType = *ptr++;
        int  bytesRemaining = rdlen - (ptr - origptr);
        if (bytesRemaining > 0) {
          _certificateAssociation.assign((const char *)ptr, bytesRemaining);
          ptr += bytesRemaining;
        }
      }
      else {
        throw out_of_range("packet too short to contain"
                           " Dwm::Dns::RRDataSMIMEA");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataSMIMEA & smimea)
    {
      if (os) {
        os << (uint16_t)smimea._certificateUsage << ' '
           << (uint16_t)smimea._selector << ' '
           << (uint16_t)smimea._matchingType << ' '
           << Base64::Encode(smimea._certificateAssociation);        
      }
      return os;
    }
    
  }  // namespace Dns

}  // namespace Dwm
