//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.1/classes/include/DwmDnsRRDataSOA.hh 10129 $
// @(#) $Id: DwmDnsRRDataSOA.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataSOA.hh
//!  \brief Dwm::Dns::RRDataSOA class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATASOA_HH_
#define _DWMDNSRRDATASOA_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates SOA resource record data.  See RFC1035 for details.
    //------------------------------------------------------------------------
    class RRDataSOA
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 6;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty MNAME and RNAME.
      //----------------------------------------------------------------------
      RRDataSOA();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c mname, @c rname, @c serial, @c refresh,
      //!  @c retry, @c expire and @c minimum.
      //----------------------------------------------------------------------
      RRDataSOA(const std::string & mname, const std::string & rname,
                uint32_t serial, uint32_t refresh, uint32_t retry,
                uint32_t expire, uint32_t minimum);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataSOA & soa) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained MNAME.
      //----------------------------------------------------------------------
      const std::string & Mname() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained MNAME.
      //----------------------------------------------------------------------
      const std::string & Mname(const std::string & mname);
      
      //----------------------------------------------------------------------
      //!  Returns the contained RNAME.
      //----------------------------------------------------------------------
      const std::string & Rname() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained RNAME.
      //----------------------------------------------------------------------
      const std::string & Rname(const std::string & rname);
      
      //----------------------------------------------------------------------
      //!  Returns the contained serial.
      //----------------------------------------------------------------------
      uint32_t Serial() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained serial.
      //----------------------------------------------------------------------
      uint32_t Serial(uint32_t serial);
      
      //----------------------------------------------------------------------
      //!  Returns the contained refresh.
      //----------------------------------------------------------------------
      uint32_t Refresh() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained refresh.
      //----------------------------------------------------------------------
      uint32_t Refresh(uint32_t refresh);
      
      //----------------------------------------------------------------------
      //!  Returns the contained retry.
      //----------------------------------------------------------------------
      uint32_t Retry() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained retry.
      //----------------------------------------------------------------------
      uint32_t Retry(uint32_t retry);
      
      //----------------------------------------------------------------------
      //!  Returns the contained expire.
      //----------------------------------------------------------------------
      uint32_t Expire() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained expire.
      //----------------------------------------------------------------------
      uint32_t Expire(uint32_t expire);
      
      //----------------------------------------------------------------------
      //!  Returns the contained minimum.
      //----------------------------------------------------------------------
      uint32_t Minimum() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained minimum.
      //----------------------------------------------------------------------
      uint32_t Minimum(uint32_t minimum);
      
      //----------------------------------------------------------------------
      //!  Encodes the SOA resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded SOA resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded SOA resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the SOA resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  SOA resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded SOA resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);
      
      //----------------------------------------------------------------------
      //!  Print the SOA resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataSOA & soa);
      
    private:
      std::string  _mname;
      std::string  _rname;
      uint32_t     _serial;
      uint32_t     _refresh;
      uint32_t     _retry;
      uint32_t     _expire;
      uint32_t     _minimum;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATANS_HH_
