//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.2.1/classes/include/DwmDnsRRDataMX.hh 10129 $
// @(#) $Id: DwmDnsRRDataMX.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataMX.hh
//!  \brief Dwm::Dns::RRDataMX class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATAMX_HH_
#define _DWMDNSRRDATAMX_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates MX resource record data.  See RFC1035 for details.
    //------------------------------------------------------------------------
    class RRDataMX
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 15;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty exchange with preference 0.
      //----------------------------------------------------------------------
      RRDataMX();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c preference and @c exchange.
      //----------------------------------------------------------------------
      RRDataMX(uint16_t preference, const std::string & exchange);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataMX & mx) const;
      
      //----------------------------------------------------------------------
      //!  Returns the preference in the MX resource record data.
      //----------------------------------------------------------------------
      uint16_t Preference() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the preference in the MX resource record data.
      //----------------------------------------------------------------------
      uint16_t Preference(uint16_t preference);
      
      //----------------------------------------------------------------------
      //!  Returns the exchange in the MX resource record data.
      //----------------------------------------------------------------------
      const std::string & Exchange() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the exchange in the MX resource record data.
      //----------------------------------------------------------------------
      const std::string & Exchange(const std::string & exchange);
      
      //----------------------------------------------------------------------
      //!  Encodes the MX resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded MX resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded MX resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the MX resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  MX resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded MX resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the MX resource record data to an ostream in human readable
      //!  form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataMX & mx);
      
    private:
      uint16_t     _preference;
      std::string  _exchange;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATAMX_HH_
