//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/src/DwmDnsResourceRecord.cc 10076 $
// @(#) $Id: DwmDnsResourceRecord.cc 10076 2018-01-25 03:50:31Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsResourceRecord.cc
//!  \brief Dwm::Dns::ResourceRecord class implementation
//---------------------------------------------------------------------------

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsLabelSequence.hh"
#include "DwmDnsRRData.hh"
#include "DwmDnsResourceRecord.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/src/DwmDnsResourceRecord.cc 10076 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    map<uint16_t,string>  ResourceRecord::_classNames = {
      { k_classIN,       "IN"      },
      { k_classCH,       "CH"      },
      { k_classHS,       "HS"      },
      { k_classUNKNOWN,  "UNKNOWN" }
    };
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    map<uint16_t,string>  ResourceRecord::_typeNames = {
      { k_typeA,          "A"          },
      { k_typeNS,         "NS"         },
      { k_typeMD,         "MD"         },
      { k_typeMF,         "MF"         },
      { k_typeCNAME,      "CNAME"      },
      { k_typeSOA,        "SOA"        },
      { k_typeMB,         "MB"         },
      { k_typeMG,         "MG"         },
      { k_typeMR,         "MR"         },
      { k_typeNULL,       "NULL"       },
      { k_typeWKS,        "WKS"        },
      { k_typePTR,        "PTR"        },
      { k_typeHINFO,      "HINFO"      },
      { k_typeMINFO,      "MINFO"      },
      { k_typeMX,         "MX"         },
      { k_typeTXT,        "TXT"        },
      { k_typeRP,         "RP"         },
      { k_typeKEY,        "KEY"        },
      { k_typeAAAA,       "AAAA"       },
      { k_typeLOC,        "LOC"        },
      { k_typeSRV,        "SRV"        },
      { k_typeCERT,       "CERT"       },
      { k_typeOPT,        "OPT"        },
      { k_typeDS,         "DS"         },
      { k_typeSSHFP,      "SSHFP"      },
      { k_typeRRSIG,      "RRSIG"      },
      { k_typeNSEC,       "NSEC"       },
      { k_typeDNSKEY,     "DNSKEY"     },
      { k_typeDHCID,      "DHCID"      },
      { k_typeNSEC3,      "NSEC3"      },
      { k_typeNSEC3PARAM, "NSEC3PARAM" },
      { k_typeTLSA,       "TLSA"       },
      { k_typeSMIMEA,     "SMIMEA"     },
      { k_typeCDS,        "CDS"        },
      { k_typeCDNSKEY,    "CDNSKEY"    },
      { k_typeOPENPGPKEY, "OPENPGPKEY" },
      { k_typeSPF,        "SPF"        },
      { k_typeURI,        "URI"        },
      { k_typeCAA,        "CAA"        }
    };

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ResourceRecord::ResourceRecord()
        : _name(), _type(k_typeUNKNOWN), _class(k_classUNKNOWN), _ttl(0),
          _rrdata()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ResourceRecord::ResourceRecord(const ResourceRecord & rr)
        : _name(rr._name), _type(rr._type), _class(rr._class), _ttl(rr._ttl),
          _rrdata(rr._rrdata)
          
    {
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ResourceRecord & ResourceRecord::operator = (const ResourceRecord & rr)
    {
      if (this != &rr) {
        _name = rr._name;
        _type = rr._type;
        _class = rr._class;
        _ttl = rr._ttl;
        _rrdata = rr._rrdata;
      }
      return *this;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ResourceRecord::operator == (const ResourceRecord & rr) const
    {
      return ((_name == rr._name)
              && (_type == rr._type)
              && (_class == rr._class)
              && (_ttl == rr._ttl)
              && (_rrdata == rr._rrdata));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & ResourceRecord::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & ResourceRecord::Name(const std::string & name)
    {
      _name = name;
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ResourceRecord::TypeEnum ResourceRecord::Type() const
    {
      return _type;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ResourceRecord::TypeEnum ResourceRecord::Type(TypeEnum typeEnum)
    {
      _type = typeEnum;
      return _type;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ResourceRecord::Class() const
    {
      return _class;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ResourceRecord::Class(uint16_t cl)
    {
      _class = cl;
      return _class;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t ResourceRecord::TTL() const
    {
      return _ttl;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t ResourceRecord::TTL(uint32_t ttl)
    {
      _ttl = ttl;
      return _ttl;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *ResourceRecord::Encode(uint8_t *pkt, uint8_t *ptr,
                                    uint16_t pktlen,
                                    LabelPositions & lps) const
    {
      LabelSequence  ls(_name);
      ptr = ls.Encode(pkt, ptr, pktlen, lps);
      uint16_t  fixedFieldLen = (sizeof(uint16_t)      // type
                                 + sizeof(uint16_t)    // class
                                 + sizeof(uint32_t)    // ttl
                                 + sizeof(uint16_t));  // rdlength
                                 
      if ((ptr + fixedFieldLen) < (pkt + pktlen)) {
        uint16_t  u16val = _type;
        u16val = htons(u16val);
        memcpy(ptr, &u16val, sizeof(u16val));
        ptr += sizeof(u16val);
        u16val = _class;
        u16val = htons(u16val);
        memcpy(ptr, &u16val, sizeof(u16val));
        ptr += sizeof(u16val);
        uint32_t  u32val = htonl(_ttl);
        memcpy(ptr, &u32val, sizeof(u32val));
        ptr += sizeof(u32val);

        uint8_t  *rdlenp = ptr;
        ptr += sizeof(uint16_t);
        uint8_t  *dataStart = ptr;
        if (_type != k_typeOPT) {
#ifdef HAVE_VARIANT
          std::visit([&](auto && arg){ptr = arg.Encode(pkt, ptr, pktlen, lps);},
                     _rrdata);
#else
          mpark::visit([&](auto && arg){ptr = arg.Encode(pkt, ptr, pktlen, lps);},
                       _rrdata);
#endif
        }
        u16val = ptr - dataStart;
        u16val = htons(u16val);
        memcpy(rdlenp, &u16val, sizeof(u16val));
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *ResourceRecord::Decode(const uint8_t *pkt,
                                          const uint8_t *ptr,
                                          uint16_t pktlen)
    {
      LabelSequence  ls;
      ptr = ls.Decode(pkt, ptr, pktlen);
      _name = (string)ls;
      uint16_t  fixedFieldLen = (sizeof(uint16_t)      // type
                                 + sizeof(uint16_t)    // class
                                 + sizeof(uint32_t)    // ttl
                                 + sizeof(uint16_t));  // rdlength
      
      if ((ptr + fixedFieldLen) < (pkt + pktlen)) {
        uint16_t  u16val;
        memcpy(&u16val, ptr, sizeof(u16val));
        _type = (TypeEnum)ntohs(u16val);
        ptr += sizeof(u16val);
        memcpy(&u16val, ptr, sizeof(u16val));
        _class = (ClassEnum)ntohs(u16val);
        ptr += sizeof(u16val);
        memcpy(&_ttl, ptr, sizeof(_ttl));
        _ttl = ntohl(_ttl);
        ptr += sizeof(_ttl);
        memcpy(&u16val, ptr, sizeof(u16val));
        u16val = ntohs(u16val);
        ptr += sizeof(u16val);
        uint16_t rdlen = u16val;

        switch (_type) {
          case k_typeA:           _rrdata = RRDataA();              break;
          case k_typeNS:          _rrdata = RRDataNS();             break;
          case k_typeCNAME:       _rrdata = RRDataCNAME();          break;
          case k_typeSOA:         _rrdata = RRDataSOA();            break;
          case k_typeMB:          _rrdata = RRDataMB();             break;
          case k_typeMG:          _rrdata = RRDataMG();             break;
          case k_typeMR:          _rrdata = RRDataMR();             break;
          case k_typePTR:         _rrdata = RRDataPTR();            break;
          case k_typeHINFO:       _rrdata = RRDataHINFO();          break;
          case k_typeMINFO:       _rrdata = RRDataMINFO();          break;
          case k_typeMX:          _rrdata = RRDataMX();             break;
          case k_typeTXT:         _rrdata = RRDataTXT();            break;
          case k_typeRP:          _rrdata = RRDataRP();             break;
          case k_typeKEY:         _rrdata = RRDataKEY();            break;
          case k_typeAAAA:        _rrdata = RRDataAAAA();           break;
          case k_typeLOC:         _rrdata = RRDataLOC();            break;
          case k_typeSRV:         _rrdata = RRDataSRV();            break;
          case k_typeCERT:        _rrdata = RRDataCERT();           break;
          case k_typeOPT:         /* no data */                     break;
          case k_typeDS:
          case k_typeCDS:         _rrdata = RRDataDS();             break;
          case k_typeSSHFP:       _rrdata = RRDataSSHFP();          break;
          case k_typeRRSIG:       _rrdata = RRDataRRSIG();          break;
          case k_typeNSEC:        _rrdata = RRDataNSEC();           break;
          case k_typeDNSKEY:
          case k_typeCDNSKEY:     _rrdata = RRDataDNSKEY();         break;
          case k_typeDHCID:       _rrdata = RRDataDHCID();          break;
          case k_typeNSEC3:       _rrdata = RRDataNSEC3();          break;
          case k_typeNSEC3PARAM:  _rrdata = RRDataNSEC3PARAM();     break;
          case k_typeTLSA:        _rrdata = RRDataTLSA();           break;
          case k_typeSMIMEA:      _rrdata = RRDataSMIMEA();         break;
          case k_typeOPENPGPKEY:  _rrdata = RRDataOPENPGPKEY();     break;
          case k_typeURI:         _rrdata = RRDataURI();            break;
          case k_typeCAA:         _rrdata = RRDataCAA();            break;
          default:                _rrdata = RRDataUnknown();        break;
        }
        if (_type != k_typeOPT) {
          ptr = visit([&](auto && arg)
                      { return arg.Decode(pkt, ptr, pktlen, rdlen); },
                      _rrdata);
        }
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ResourceRecord::ClassName(uint16_t cl)
    {
      string  rc;
      auto  i = _classNames.find(cl);
      if (i != _classNames.end()) {
        rc = i->second;
      }
      else {
        rc = to_string(cl);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string ResourceRecord::TypeName(uint16_t rrtype)
    {
      string  rc;
      auto i = _typeNames.find(rrtype);
      if (i != _typeNames.end()) {
        rc = i->second;
      }
      else {
        rc = to_string(rrtype);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & ResourceRecord::PrintData(ostream & os) const
    {
#ifdef HAVE_VARIANT
      std::visit([&](auto && arg){os << arg;}, _rrdata);
#else
      mpark::visit([&](auto && arg){os << arg;}, _rrdata);
#endif
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const ResourceRecord & rr)
    {
      if (os) {
        os << rr._name << ' ' << rr._ttl << ' '
           << ResourceRecord::ClassName(rr._class) << ' '
           << ResourceRecord::TypeName(rr._type) << ' ';
        rr.PrintData(os);
      }
      return os;
    }
    

  }  // namespace Dns

}  // namespace Dwm
