//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/src/DwmDnsRRDataNSEC.cc 10133 $
// @(#) $Id: DwmDnsRRDataNSEC.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataNSEC.cc
//!  \brief Dwm::Dns::RRDataNSEC class implementation
//---------------------------------------------------------------------------

#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsLabelSequence.hh"
#include "DwmDnsResourceRecord.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/src/DwmDnsRRDataNSEC.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataNSEC::RRDataNSEC()
        : _nextDomainName(), _bitmaps()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataNSEC::RRDataNSEC(const std::string & nextDomainName,
                           const std::vector<NSECBitmap> & bitmaps)
        : _nextDomainName(nextDomainName), _bitmaps(bitmaps)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataNSEC::operator == (const RRDataNSEC & nsec) const
    {
      return ((_nextDomainName == nsec._nextDomainName)
              && (_bitmaps == nsec._bitmaps));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RRDataNSEC::NextDomainName() const
    {
      return _nextDomainName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string &
    RRDataNSEC::NextDomainName(const std::string & domainName)
    {
      _nextDomainName = domainName;
      return _nextDomainName;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<NSECBitmap> & RRDataNSEC::Bitmaps() const
    {
      return _bitmaps;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<NSECBitmap> &
    RRDataNSEC::Bitmaps(const vector<NSECBitmap> & bitmaps)
    {
      _bitmaps = bitmaps;
      return _bitmaps;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataNSEC::Encode(uint8_t *pkt, uint8_t *ptr,
                                uint16_t pktlen, LabelPositions & lps) const
    {
      LabelSequence  ls(_nextDomainName);
      ptr = ls.Encode(pkt, ptr, pktlen);
      for (auto & bitmap : _bitmaps) {
        ptr = bitmap.Encode(pkt, ptr, pktlen);
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataNSEC::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                      uint16_t pktlen, uint16_t rdlen)
    {
      _nextDomainName.clear();
      _bitmaps.clear();

      const uint8_t  *origptr = ptr;
      int             bytesRemaining = rdlen;
      
      LabelSequence  ls;
      ptr = ls.Decode(pkt, ptr, pktlen);
      _nextDomainName = (string)ls;
      bytesRemaining -= (ptr - origptr);
      while (bytesRemaining > 0) {
        const uint8_t  *bmptr = ptr;
        NSECBitmap  bitmap;
        ptr = bitmap.Decode(pkt, ptr, pktlen, bytesRemaining);
        bytesRemaining -= (ptr - bmptr);
        _bitmaps.push_back(bitmap);
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  See RFC4034 for details.
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataNSEC & nsec)
    {
      if (os) {
        os << nsec._nextDomainName;
        for (auto & bm : nsec._bitmaps) {
          os << bm;
        }
      }
      return os;
    }
    
    
  }  // namespace Dns

}  // namespace Dwm
