//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/include/DwmDnsRRDataURI.hh 10129 $
// @(#) $Id: DwmDnsRRDataURI.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataURI.hh
//!  \brief Dwm::Dns::RRDataURI class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATAURI_HH_
#define _DWMDNSRRDATAURI_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates URI resource record data.  See RFC7553 for details.
    //------------------------------------------------------------------------
    class RRDataURI
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 256;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty target.
      //----------------------------------------------------------------------
      RRDataURI();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c priority, @c weight and @c target.
      //----------------------------------------------------------------------
      RRDataURI(uint16_t priority, uint16_t weight,
                const std::string & target);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataURI & srv) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained priority.
      //----------------------------------------------------------------------
      uint16_t Priority() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained priority.
      //----------------------------------------------------------------------
      uint16_t Priority(uint16_t priority);
      
      //----------------------------------------------------------------------
      //!  Returns the contained weight.
      //----------------------------------------------------------------------
      uint16_t Weight() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained weight.
      //----------------------------------------------------------------------
      uint16_t Weight(uint16_t weight);
      
      //----------------------------------------------------------------------
      //!  Returns the contained target.
      //----------------------------------------------------------------------
      const std::string & Target() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained target.
      //----------------------------------------------------------------------
      const std::string & Target(const std::string & target);
      
      //----------------------------------------------------------------------
      //!  Encodes the URI resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded URI resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded URI resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the URI resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  URI resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded URI resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);
      
      //----------------------------------------------------------------------
      //!  Print the URI resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataURI & uri);
      
    private:
      uint16_t     _priority;
      uint16_t     _weight;
      std::string  _target;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATAURI_HH_
