//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/include/DwmDnsRRDataKEY.hh 10129 $
// @(#) $Id: DwmDnsRRDataKEY.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataKEY.hh
//!  \brief Dwm::Dns::RRDataKEY class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATAKEY_HH_
#define _DWMDNSRRDATAKEY_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates KEY resource record data.
    //!  See RFC 2535 for details.  Note that RFC 2535 is basically dead
    //!  with respect to DNSSEC, so no one should be using it for DNSSEC
    //!  purposes, and probably shouldn't be using it at all.
    //------------------------------------------------------------------------
    class RRDataKEY
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 25;
      
      static const uint16_t  k_flagsACMask      = 0xC000;
      static const uint16_t  k_flagsXTMask      = 0x1000;
      static const uint16_t  k_flagsNAMTYPEMask = 0x0300;
      static const uint16_t  k_flagsSIGMask     = 0x000F;

      static const uint8_t   k_protocolReserved = 0;
      static const uint8_t   k_protocolTLS      = 1;
      static const uint8_t   k_protocolEmail    = 2;
      static const uint8_t   k_protocolDNSSEC   = 3;
      static const uint8_t   k_protocolIPSEC    = 4;
      static const uint8_t   k_protocolAll      = 255;

      static const uint8_t   k_algorithmReserved       = 0;
      static const uint8_t   k_algorithmRSA_MD5        = 1;
      static const uint8_t   k_algorithmDiffieHellman  = 2;
      static const uint8_t   k_algorithmDSA            = 3;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty public key.
      //----------------------------------------------------------------------
      RRDataKEY();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c flags, @c protocol, @c algorithm,
      //!  @c moreFlags and @c publicKey.
      //----------------------------------------------------------------------
      RRDataKEY(uint16_t flags, uint8_t protocol, uint8_t algorithm,
                uint16_t moreFlags, const std::string & publicKey);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataKEY & key) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained flags.
      //----------------------------------------------------------------------
      uint16_t Flags() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained flags.
      //----------------------------------------------------------------------
      uint16_t Flags(uint16_t flags);
      
      //----------------------------------------------------------------------
      //!  Returns the contained protocol.
      //----------------------------------------------------------------------
      uint8_t Protocol() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained protocol.
      //----------------------------------------------------------------------
      uint8_t Protocol(uint8_t protocol);
      
      //----------------------------------------------------------------------
      //!  Returns the contained algorithm.
      //----------------------------------------------------------------------
      uint8_t Algorithm() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained algorithm.
      //----------------------------------------------------------------------
      uint8_t Algorithm(uint8_t algorithm);
      
      //----------------------------------------------------------------------
      //!  Returns the contained 'more flags'.
      //----------------------------------------------------------------------
      uint16_t MoreFlags() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained 'more flags'.
      //----------------------------------------------------------------------
      uint16_t MoreFlags(uint16_t moreFlags);
      
      //----------------------------------------------------------------------
      //!  Returns the contained public key.
      //----------------------------------------------------------------------
      const std::string & PublicKey() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained public key.
      //----------------------------------------------------------------------
      const std::string & PublicKey(const std::string & publicKey);
      
      //----------------------------------------------------------------------
      //!  Encodes the KEY resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded KEY resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded KEY resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the KEY resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  KEY resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded KEY resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the KEY resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataKEY & key);
      
    private:
      uint16_t     _flags;
      uint8_t      _protocol;
      uint8_t      _algorithm;
      uint16_t     _moreFlags;
      std::string  _publicKey;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATAKEY_HH_
