//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/include/DwmDnsRRDataHINFO.hh 10129 $
// @(#) $Id: DwmDnsRRDataHINFO.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataHINFO.hh
//!  \brief Dwm::Dns::RRDataHINFO class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATAHINFO_HH_
#define _DWMDNSRRDATAHINFO_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates HINFO resource record data.  See RFC1035 for details.
    //------------------------------------------------------------------------
    class RRDataHINFO
      : public RRData
    {
    public:
      static const uint16_t k_rrtype = 13;
      
      //----------------------------------------------------------------------
      //!  Construct empty HINFO resource record data.
      //----------------------------------------------------------------------
      RRDataHINFO();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c cpu and operating system @c os.
      //----------------------------------------------------------------------
      RRDataHINFO(const std::string & cpu, const std::string & os);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataHINFO & hinfo) const;
      
      //----------------------------------------------------------------------
      //!  Returns the CPU in the HINFO resource record data.
      //----------------------------------------------------------------------
      const std::string & Cpu() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the CPU in the HINFO resource record data.
      //----------------------------------------------------------------------
      const std::string & Cpu(const std::string & cpu);
      
      //----------------------------------------------------------------------
      //!  Returns the operating system in the HINFO resource record data.
      //----------------------------------------------------------------------
      const std::string & Os() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the operating system in the HINFO resource
      //!  record data.
      //----------------------------------------------------------------------
      const std::string & Os(const std::string & os);
      
      //----------------------------------------------------------------------
      //!  Encodes the HINFO resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded HINFO resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded HINFO resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the HINFO resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  HINFO resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded HINFO resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the HINFO resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataHINFO & hinfo);
      
    private:
      std::string  _cpu;
      std::string  _os;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATAHINFO_HH_
