//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/include/DwmDnsMessageHeader.hh 10136 $
// @(#) $Id: DwmDnsMessageHeader.hh 10136 2018-01-28 06:01:09Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsMessageHeader.hh
//!  \brief Dwm::Dns::MessageHeader class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSMESSAGEHEADER_HH_
#define _DWMDNSMESSAGEHEADER_HH_

#include <cstdint>

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup messagegroup
    //!  Encapsulates a DNS message header.  See RFC1035 for details.
    //------------------------------------------------------------------------
    class MessageHeader
    {
    public:
      //----------------------------------------------------------------------
      //!  constructor
      //----------------------------------------------------------------------
      MessageHeader();
      
      //----------------------------------------------------------------------
      //!  Equal-to operator.  Mostly here for unit tests.
      //----------------------------------------------------------------------
      bool operator == (const MessageHeader & messageHeader) const;
      
      //----------------------------------------------------------------------
      //!  Clears the message header.
      //----------------------------------------------------------------------
      void Clear();
      
      //----------------------------------------------------------------------
      //!  Returns the message ID.
      //----------------------------------------------------------------------
      uint16_t Id() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the message ID.
      //----------------------------------------------------------------------
      uint16_t Id(uint16_t id);
      
      //----------------------------------------------------------------------
      //!  Returns true if the message is a response, false if the message
      //!  is a query.
      //----------------------------------------------------------------------
      bool IsResponse() const;
      
      //----------------------------------------------------------------------
      //!  If @s isResponse is @c true, sets the message to be a response.
      //!  Else sets the message to be a query.
      //----------------------------------------------------------------------
      bool IsResponse(bool isResponse);
      
      //----------------------------------------------------------------------
      //!  Returns the OPCODE.
      //----------------------------------------------------------------------
      uint8_t OpCode() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the OPCODE.
      //----------------------------------------------------------------------
      uint8_t OpCode(uint8_t opcode);
      
      //----------------------------------------------------------------------
      //!  Returns true if the message is authoritative.  Only valid for
      //!  responses.
      //----------------------------------------------------------------------
      bool IsAuthoritativeAnswer() const;
      
      //----------------------------------------------------------------------
      //!  If @c isAuthoritative is @c true, sets a response message as
      //!  authoritative.
      //----------------------------------------------------------------------
      bool IsAuthoritativeAnswer(bool isAuthoritative);
      
      //----------------------------------------------------------------------
      //!  Returns true if the message was truncated.
      //----------------------------------------------------------------------
      bool IsTruncated() const;
      
      //----------------------------------------------------------------------
      //!  If @c isTruncated is true, set the truncated bit in the header,
      //!  indicating the message (a response) was truncated.
      //----------------------------------------------------------------------
      bool IsTruncated(bool isTruncated);
      
      //----------------------------------------------------------------------
      //!  Returns true if recursion is desired.
      //----------------------------------------------------------------------
      bool RecursionDesired() const;
      
      //----------------------------------------------------------------------
      //!  Sets the recursion desired bit in the header.  If recursion is
      //!  desired, @c recursionDesired should be @c true.
      //----------------------------------------------------------------------
      bool RecursionDesired(bool recursionDesired);
      
      //----------------------------------------------------------------------
      //!  Returns the recursion available bit in the header.
      //----------------------------------------------------------------------
      bool RecursionAvailable() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the recursion available bit in the header.
      //----------------------------------------------------------------------
      bool RecursionAvailable(bool recursionAvailable);
      
      uint8_t Z() const;
      uint8_t Z(uint8_t z);

      //----------------------------------------------------------------------
      //!  Returns the response code.
      //----------------------------------------------------------------------
      uint8_t ResponseCode() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the response code.
      //----------------------------------------------------------------------
      uint8_t ResponseCode(uint8_t responseCode);
      
      //----------------------------------------------------------------------
      //!  Returns the question count.
      //----------------------------------------------------------------------
      uint16_t QuestionCount() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the question count.
      //----------------------------------------------------------------------
      uint16_t QuestionCount(uint16_t questionCount);
      
      //----------------------------------------------------------------------
      //!  Returns the answer count, i.e. the number of records in the
      //!  authority section of the message.
      //----------------------------------------------------------------------
      uint16_t AnswerCount() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the answer count, i.e. the number of records in
      //!  the authority section of the message.
      //----------------------------------------------------------------------
      uint16_t AnswerCount(uint16_t answerCount);
      
      //----------------------------------------------------------------------
      //!  Returns the authority count, i.e. the number of resource records
      //!  in the authority sectino of the message.
      //----------------------------------------------------------------------
      uint16_t AuthorityCount() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the authority count, i.e. the number of resource
      //!  records in the authority sectino of the message.
      //----------------------------------------------------------------------
      uint16_t AuthorityCount(uint16_t authorityCount);
      
      //----------------------------------------------------------------------
      //!  Returns the additional count, i.e. the number of resource records
      //!  in the additional section of the message.
      //----------------------------------------------------------------------
      uint16_t AdditionalCount() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the additional count, i.e. the number of resource
      //!  records in the additional section of the message.
      //----------------------------------------------------------------------
      uint16_t AdditionalCount(uint16_t additionalCount);
      
      //----------------------------------------------------------------------
      //!  Encodes the header for transport.  @c pkt must point to the start
      //!  of a buffer of length @c pktlen, and the encoded header will be
      //!  placed at @c ptr.  Returns the address just beyond the end of
      //!  the encoded header in @c buf on success.  Throws std::out_of_range
      //!  if there is not enough space in @c pkt to contain the encoded
      //!  header.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the encoded header starting at @c ptr inside a buffer
      //!  @c pkt of length @c pktlen.  Returns the address jsut beyond the
      //!  encoded header in @c pkt on success.  Throws std::out_of_range
      //!  if @c pkt was too short to contain an encoded header.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen);
      
    private:
      typedef struct __attribute__((packed)) {
        uint16_t  id;
        uint16_t  flags;
        uint16_t  questionCount;
        uint16_t  answerCount;
        uint16_t  authorityCount;
        uint16_t  additionalCount;
      } data_t;

      data_t  _data;

      static void ToHostByteOrder(data_t & data);
      static void ToNetworkByteOrder(data_t & data);
    };

  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSMESSAGEHEADER_HH_
