//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.1/classes/include/DwmDnsMessage.hh 10136 $
// @(#) $Id: DwmDnsMessage.hh 10136 2018-01-28 06:01:09Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsMessage.hh
//!  \brief Dwm::Dns::Message class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSMESSAGE_HH_
#define _DWMDNSMESSAGE_HH_

extern "C" {
  #include <unistd.h>
}

#include <cstdint>
#include <vector>

#include "DwmDnsMessageHeader.hh"
#include "DwmDnsMessageQuestion.hh"
#include "DwmDnsResourceRecord.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @defgroup messagegroup Top level message classes
    //------------------------------------------------------------------------
    
    //------------------------------------------------------------------------
    //!  @ingroup messagegroup
    //!  Encapsulates a DNS message.  See RFC1035 for details.
    //------------------------------------------------------------------------
    class Message
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructor.
      //----------------------------------------------------------------------
      Message();
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly useful for unit tests.
      //----------------------------------------------------------------------
      bool operator == (const Message & message) const;
      
      //----------------------------------------------------------------------
      //!  Clears the message contents.
      //----------------------------------------------------------------------
      void Clear();
      
      //----------------------------------------------------------------------
      //!  Returns a const reference to the message header.
      //----------------------------------------------------------------------
      const MessageHeader & Header() const
      {
        return _header;
      }
      
      //----------------------------------------------------------------------
      //!  Returns a mutable reference to the message header.
      //----------------------------------------------------------------------
      MessageHeader & Header()
      {
        return _header;
      }

      //----------------------------------------------------------------------
      //!  Returns a const reference to the questions in the message.
      //----------------------------------------------------------------------
      const std::vector<MessageQuestion> & Questions() const
      {
        return _questions;
      }

      //----------------------------------------------------------------------
      //!  Returns a mutable reference to the questions in the message.
      //----------------------------------------------------------------------
      std::vector<MessageQuestion> & Questions()
      {
        return _questions;
      }
      
      //----------------------------------------------------------------------
      //!  Returns a const reference to the records from the answers
      //!  section of the message.
      //----------------------------------------------------------------------
      const std::vector<ResourceRecord> & Answers() const
      {
        return _answers;
      }
      
      //----------------------------------------------------------------------
      //!  Returns a mutable reference to the records from the answers
      //!  section of the message.
      //----------------------------------------------------------------------
      std::vector<ResourceRecord> & Answers()
      {
        return _answers;
      }
        
      //----------------------------------------------------------------------
      //!  Returns a const reference to the records from the authority
      //!  section of the message.
      //----------------------------------------------------------------------
      const std::vector<ResourceRecord> & Authorities() const
      {
        return _authorities;
      }
      
      //----------------------------------------------------------------------
      //!  Returns a mutable reference to the records from the authority
      //!  section of the message.
      //----------------------------------------------------------------------
      std::vector<ResourceRecord> & Authorities()
      {
        return _authorities;
      }
      
      //----------------------------------------------------------------------
      //!  Returns a const reference to the records from the additional
      //!  section of the message.
      //----------------------------------------------------------------------
      const std::vector<ResourceRecord> & Additional() const
      {
        return _additional;
      }
      
      //----------------------------------------------------------------------
      //!  Returns a mutable reference to the records from the additional
      //!  section of the message.
      //----------------------------------------------------------------------
      std::vector<ResourceRecord> & Additional()
      {
        return _additional;
      }
      
      //----------------------------------------------------------------------
      //!  Encode the Message in @c pkt of length @c pktlen, starting at
      //!  @c ptr.  Returns the address of the first byte after the encoded
      //!  message on success.  Throws std::out_of_range if the message
      //!  will not fit in the packet.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the Message in @c pkt of length @c pktlen, starting at
      //!  @c ptr.  Returns the address of the first byte in @c pkt after
      //!  the encoded message on success.  Throws std::out_of_range if
      //!  @c pkt was too short to hold the message or another decoding
      //!  error occurred.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen);

      //----------------------------------------------------------------------
      //!  A wrapper for sendto().  Sends the message to the destination
      //!  specified destination 'to' via socket descriptor 'sockFd'.
      //!  'flags' has the same meaning as flags for sendto(), and 'tolen'
      //!  should be the size of 'to'.  Returns the return value of sendto().
      //----------------------------------------------------------------------
      ssize_t SendTo(int sockFd, int flags, const struct sockaddr *to,
                     int tolen) const;

      //----------------------------------------------------------------------
      //!  Writes the message to a connected TCP socket @c fd.  Note that
      //!  under the hood, we write the 2-byte length before writing the
      //!  Message content, since that's how DNS over TCP works per RFC1035.
      //!  Returns the number of bytes written on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Write(int fd) const;
      
      //----------------------------------------------------------------------
      //!  A wrapper for recvfrom().  Receives the message from socket
      //!  descriptor 'sockFd'.  'from' and 'to' will be filled with
      //!  information about the source of the message (just like
      //!  recvfrom()).  'flags' has the same meaning as flags for
      //!  recvfrom().  Returns the return value of recvfrom() (which is
      //!  called under the hood).
      //----------------------------------------------------------------------
      ssize_t RecvFrom(int sockFd, int flags, sockaddr *from,
                       socklen_t *fromlen);

      //----------------------------------------------------------------------
      //!  Reads the Message from a connected TCP socket @c fd.  Note that
      //!  under the hood, we read the 2-byte length before reading the
      //!  Message content, since that's how DNS over TCP works per RFC1035.
      //!  Returns the number of bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Read(int fd);
      
      //----------------------------------------------------------------------
      //!  Enable EDNS by setting the maximum packet size to @c pduSize
      //!  and enable or disable DNSSEC.
      //----------------------------------------------------------------------
      void EnableEDNS(uint16_t pduSize, bool doDnssec);
      
    private:
      mutable MessageHeader         _header;
      std::vector<MessageQuestion>  _questions;
      std::vector<ResourceRecord>   _answers;
      std::vector<ResourceRecord>   _authorities;
      std::vector<ResourceRecord>   _additional;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSMESSAGE_HH_
