//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.0/classes/src/DwmDnsRRDataDHCID.cc 10133 $
// @(#) $Id: DwmDnsRRDataDHCID.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataDHCID.cc
//!  \brief Dwm::Dns::RRDataDHCID class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmBase64.hh"
#include "DwmDnsRRDataDHCID.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.0/classes/src/DwmDnsRRDataDHCID.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataDHCID::RRDataDHCID()
        : _id(0), _digestType(0), _digest()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataDHCID::RRDataDHCID(uint16_t id, uint8_t digestType,
                             const string & digest)
        : _id(id), _digestType(digestType), _digest(digest)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataDHCID::operator == (const RRDataDHCID & dhcid) const
    {
      return ((_id == dhcid._id)
              && (_digestType == dhcid._digestType)
              && (_digest == dhcid._digest));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataDHCID::Id() const
    {
      return _id;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataDHCID::Id(uint16_t id)
    {
      _id = id;
      return _id;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t RRDataDHCID::DigestType() const
    {
      return _digestType;
    }
    
    uint8_t RRDataDHCID::DigestType(uint8_t digestType)
    {
      _digestType = digestType;
      return _digestType;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataDHCID::Digest() const
    {
      return _digest;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataDHCID::Digest(const string & digest)
    {
      _digest = digest;
      return _digest;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataDHCID::Encode(uint8_t *pkt, uint8_t *ptr,
                                 uint16_t pktlen, LabelPositions & lps) const
    {
      if ((ptr + 3) <= (pkt + pktlen)) {
        uint16_t  val = htons(_id);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        *ptr++ = _digestType;
        if (! _digest.empty()) {
          if ((ptr + _digest.size()) <= (pkt + pktlen)) {
            memcpy(ptr, _digest.c_str(), _digest.size());
            ptr += _digest.size();
          }
          else {
            throw out_of_range("Dns::RRDataDHCID will not fit in packet");
          }
        }
      }
      else {
        throw out_of_range("Dns::RRDataDHCID will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataDHCID::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                       uint16_t pktlen, uint16_t rdlen)
    {
      _id = 0;
      _digestType = 0;
      _digest.clear();
      
      const uint8_t  *origptr = ptr;
      if ((rdlen > 2) && ((ptr + 3) <= (pkt + pktlen))) {
        memcpy(&_id, ptr, sizeof(_id));
        _id = ntohs(_id);
        ptr += sizeof(_id);
        _digestType = *ptr++;
        int  bytesRemaining = rdlen - (ptr - origptr);
        if (bytesRemaining > 0) {
          _digest.assign((const char *)ptr, bytesRemaining);
          ptr += bytesRemaining;
        }
      }
      else {
        throw out_of_range("packet too short to contain Dns::RRDataDHCID");
      }
      return ptr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataDHCID & dhcid)
    {
      if (os) {
        os << dhcid._id << ' ' << (uint16_t)dhcid._digestType << ' '
           << Base64::Encode(dhcid._digest);
      }
      return os;
    }
    
    
  }  // namespace Dns

}  // namespace Dwm
