//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.0/classes/include/DwmDnsRRDataSSHFP.hh 10129 $
// @(#) $Id: DwmDnsRRDataSSHFP.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataSSHFP.hh
//!  \brief Dwm::Dns::RRDataSSHFP class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATASSHFP_HH_
#define _DWMDNSRRDATASSHFP_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates SSHFP recource record data.  See RFC4255 for details.
    //------------------------------------------------------------------------
    class RRDataSSHFP
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 44;
      
      //  algorithms (RFC4255 section 3.1.1)
      static const uint8_t  k_algorithmReserved    = 0;
      static const uint8_t  k_algorithmRSA         = 1;
      static const uint8_t  k_algorithmDSS         = 2;

      //  fingerprint types (RFC4255 section 3.1.2)
      static const uint8_t  k_fpTypeReserved  = 0;
      static const uint8_t  k_fpTypeSHA1      = 1;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty fingerprint.
      //----------------------------------------------------------------------
      RRDataSSHFP();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c algorithm, @c fpType and
      //!  @c fingerprint.
      //----------------------------------------------------------------------
      RRDataSSHFP(uint8_t algorithm, uint8_t fpType,
                  const std::string & fingerprint);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataSSHFP & sshfp) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained algorithm.
      //----------------------------------------------------------------------
      uint8_t Algorithm() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained algorithm.
      //----------------------------------------------------------------------
      uint8_t Algorithm(uint8_t algorithm);
      
      //----------------------------------------------------------------------
      //!  Returns the contained fingerprint type.
      //----------------------------------------------------------------------
      uint8_t FPType() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained fingerprint type.
      //----------------------------------------------------------------------
      uint8_t FPType(uint8_t fpType);
      
      //----------------------------------------------------------------------
      //!  Returns the contained fingerprint.
      //----------------------------------------------------------------------
      const std::string & Fingerprint() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained fingerprint.
      //----------------------------------------------------------------------
      const std::string & Fingerprint(const std::string & fingerprint);
      
      //----------------------------------------------------------------------
      //!  Encodes the SSHFP resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded SSHFP resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded SSHFP resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the SSHFP resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  SSHFP resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded SSHFP resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);
      
      //----------------------------------------------------------------------
      //!  Print the SSHFP resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataSSHFP & sshfp);
      
    private:
      uint8_t      _algorithm;
      uint8_t      _fpType;
      std::string  _fingerprint;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATASSHFP_HH_
