//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.0/classes/include/DwmDnsRRDataRP.hh 10129 $
// @(#) $Id: DwmDnsRRDataRP.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataRP.hh
//!  \brief Dwm::Dns::RRDataRP class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATARP_HH_
#define _DWMDNSRRDATARP_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates RP resource record data.  See RFC1183 for details.
    //------------------------------------------------------------------------
    class RRDataRP
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 17;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty MBOX and empty TXT.
      //----------------------------------------------------------------------
      RRDataRP();
      
      //----------------------------------------------------------------------
      //!  Construct with the given @c mbox and @c txt.
      //----------------------------------------------------------------------
      RRDataRP(const std::string & mbox, const std::string & txt);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataRP & rp) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained mbox.
      //----------------------------------------------------------------------
      const std::string & Mbox() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained mbox.
      //----------------------------------------------------------------------
      const std::string & Mbox(const std::string & mbox);
      
      //----------------------------------------------------------------------
      //!  Returns the contained txt.
      //----------------------------------------------------------------------
      const std::string & Txt() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained txt.
      //----------------------------------------------------------------------
      const std::string & Txt(const std::string & txt);
      
      //----------------------------------------------------------------------
      //!  Encodes the RP resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded RP resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded RP resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the RP resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  RP resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded RP resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);
      
      //----------------------------------------------------------------------
      //!  Print the RP resource record data to an ostream in human readable
      //!  form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataRP & rp);
      
    private:
      std::string  _mbox;
      std::string  _txt;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATARP_HH_
