//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.0/classes/include/DwmDnsRRDataLOC.hh 10138 $
// @(#) $Id: DwmDnsRRDataLOC.hh 10138 2018-01-28 07:39:44Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataLOC.hh
//!  \brief Dwm::Dns::RRDataLOC class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATALOC_HH_
#define _DWMDNSRRDATALOC_HH_

#include <cstdint>
#include <iostream>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {
    
    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates LOC resource record data.
    //------------------------------------------------------------------------
    class RRDataLOC
      : public RRData
    {
    public:
      static const uint16_t k_rrtype = 29;
      
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      RRDataLOC();

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      RRDataLOC(uint8_t version, uint8_t size,
                uint8_t horizontalPrecision, uint8_t verticalPrecision,
                uint32_t iatitude, uint32_t longitude, uint32_t altitude);

      bool operator == (const RRDataLOC & loc) const;

      //----------------------------------------------------------------------
      //!  Returns the version field.
      /*!  From RFC1876:
       *
       *   Version number of the representation.  This must be zero.
       *   Implementations are required to check this field and make no
       *   assumptions about the format of unrecognized versions.
       */
      //----------------------------------------------------------------------
      uint8_t Version() const;
    
      //----------------------------------------------------------------------
      //!  Sets and returns the version field.
      /*!  From RFC1876:
       *
       *   Version number of the representation.  This must be zero.
       *   Implementations are required to check this field and make no
       *   assumptions about the format of unrecognized versions.
       */
      //----------------------------------------------------------------------
      uint8_t Version(uint8_t version);
    
      //----------------------------------------------------------------------
      //!  Returns the size field.
      /*!  From RFC1876:
       *
       *   The diameter of a sphere enclosing the described entity, in
       *   centimeters, expressed as a pair of four-bit unsigned integers,
       *   each ranging from zero to nine, with the most significant four
       *   bits representing the base and the second number representing
       *   the power of ten by which to multiply the base.  This allows
       *   sizes from 0e0 (<1cm) to 9e9 (90,000km) to be expressed.  This
       *   representation was chosen such that the hexadecimal
       *   representation can be read by eye; 0x15 = 1e5.  Four-bit values
       *   greater than 9 are undefined, as are values with a base of zero
       *   and a non-zero exponent.
       *
       *   Since 20000000m (represented by the value 0x29) is greater than
       *   the equatorial diameter of the WGS 84 ellipsoid (12756274m), it
       *   is therefore suitable for use as a "worldwide" size.
       */
      //----------------------------------------------------------------------
      uint8_t Size() const;
    
      //----------------------------------------------------------------------
      //!  Sets and returns the size field.
      /*!  From RFC1876:
       *
       *   The diameter of a sphere enclosing the described entity, in
       *   centimeters, expressed as a pair of four-bit unsigned integers,
       *   each ranging from zero to nine, with the most significant four
       *   bits representing the base and the second number representing
       *   the power of ten by which to multiply the base.  This allows
       *   sizes from 0e0 (<1cm) to 9e9 (90,000km) to be expressed.  This
       *   representation was chosen such that the hexadecimal
       *   representation can be read by eye; 0x15 = 1e5.  Four-bit values
       *   greater than 9 are undefined, as are values with a base of zero
       *   and a non-zero exponent.
       *
       *   Since 20000000m (represented by the value 0x29) is greater than
       *   the equatorial diameter of the WGS 84 ellipsoid (12756274m), it
       *   is therefore suitable for use as a "worldwide" size.
       */
      //----------------------------------------------------------------------
      uint8_t Size(uint8_t size);
    
      //----------------------------------------------------------------------
      //!  Returns the horizontal precision field.
      /*!  From RFC1876:
       *   The horizontal precision of the data, in centimeters,
       *   expressed using the same representation as SIZE.  This is
       *   the diameter of the horizontal "circle of error", rather
       *   than a "plus or minus" value.  (This was chosen to match
       *   the interpretation of SIZE; to get a "plus or minus" value,
       *   divide by 2.)
       */
      //----------------------------------------------------------------------
      uint8_t HorizontalPrecision() const;
    
      //----------------------------------------------------------------------
      //!  Sets and returns the horizontal precision field.
      /*!  From RFC1876:
       *   The horizontal precision of the data, in centimeters,
       *   expressed using the same representation as SIZE.  This is
       *   the diameter of the horizontal "circle of error", rather
       *   than a "plus or minus" value.  (This was chosen to match
       *   the interpretation of SIZE; to get a "plus or minus" value,
       *   divide by 2.)
       */
      //----------------------------------------------------------------------
      uint8_t HorizontalPrecision(uint8_t horizontalPrecision);
    
      //----------------------------------------------------------------------
      //!  Returns the vertical precision field.
      /*!  From RFC1876:
       *   The vertical precision of the data, in centimeters, expressed using
       *   the sane representation as for SIZE.  This is the total potential
       *   vertical error, rather than a "plus or minus" value.  (This was
       *   chosen to match the interpretation of SIZE; to get a "plus or minus"
       *   value, divide by 2.)  Note that if altitude above or below sea level
       *   is used as an approximation for altitude relative to the [WGS 84]
       *   ellipsoid, the precision value should be adjusted.
       */
      //----------------------------------------------------------------------
      uint8_t VerticalPrecision() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the vertical precision field.
      /*!  From RFC1876:
       *
       *   The vertical precision of the data, in centimeters, expressed using
       *   the sane representation as for SIZE.  This is the total potential
       *   vertical error, rather than a "plus or minus" value.  (This was
       *   chosen to match the interpretation of SIZE; to get a "plus or minus"
       *   value, divide by 2.)  Note that if altitude above or below sea level
       *   is used as an approximation for altitude relative to the [WGS 84]
       *   ellipsoid, the precision value should be adjusted.
       */
      //----------------------------------------------------------------------
      uint8_t VerticalPrecision(uint8_t verticalPrecision);
      
      //----------------------------------------------------------------------
      //!  Returns the latitude field.
      /*!  From RFC1876:
       *
       *   The latitude of the center of the sphere described by the SIZE
       *   field, expressed as a 32-bit integer, most significant octet first
       *   (network standard byte order), in thousandths of a second of arc.
       *   2^31 represents the equator; numbers above that are north latitude.
       */
      //----------------------------------------------------------------------
      uint32_t Latitude() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the latitude field.
      /*!  From RFC1876:
       *
       *   The latitude of the center of the sphere described by the SIZE
       *   field, expressed as a 32-bit integer, most significant octet first
       *   (network standard byte order), in thousandths of a second of arc.
       *   2^31 represents the equator; numbers above that are north latitude.
       */
      //----------------------------------------------------------------------
      uint32_t Latitude(uint32_t latitude);
    
      //----------------------------------------------------------------------
      //!  Returns the longitude field.
      /*!  From RFC1876:
       *
       *   The longitude of the center of the sphere described by the SIZE
       *   field, expressed as a 32-bit integer, most significant octet first
       *   (network standard byte order), in thousandths of a second of arc,
       *   rounded away from the prime meridian.  2^31 represents the prime
       *   meridian; numbers above that are east longitude.
       */                            
      //----------------------------------------------------------------------
      uint32_t Longitude() const;
    
      //----------------------------------------------------------------------
      //!  Sets and returns the longitude field.
      /*!  From RFC1876:
       *
       *   The longitude of the center of the sphere described by the SIZE
       *   field, expressed as a 32-bit integer, most significant octet first
       *   (network standard byte order), in thousandths of a second of arc,
       *   rounded away from the prime meridian.  2^31 represents the prime
       *   meridian; numbers above that are east longitude.
       */                            
      //----------------------------------------------------------------------
      uint32_t Longitude(uint32_t longitude);
      
      //----------------------------------------------------------------------
      //!  Returns the altitude field.
      /*!  From RFC1876:
       *
       *   The altitude of the center of the sphere described by the SIZE
       *   field, expressed as a 32-bit integer, most significant octet first
       *   (network standard byte order), in centimeters, from a base of
       *   100,000m below the [WGS 84] reference spheroid used by GPS
       *   (semimajor axis a=6378137.0, reciprocal flattening
       *   rf=298.257223563).
       *   Altitude above (or below) sea level may be used as an approximation
       *   of altitude relative to the the [WGS 84] spheroid, though due to the
       *   Earth's surface not being a perfect spheroid, there will be
       *   differences.  (For example, the geoid (which sea level approximates)
       *   for the continental US ranges from 10 meters to 50 meters below the
       *   [WGS 84] spheroid.  Adjustments to ALTITUDE and/or VERT PRE will be
       *   necessary in most cases.  The Defense Mapping Agency publishes geoid
       *   height values relative to the [WGS 84] ellipsoid.
       */
      //----------------------------------------------------------------------
      uint32_t Altitude() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the altitude field.
      /*!  From RFC1876:
       *
       *   The altitude of the center of the sphere described by the SIZE
       *   field, expressed as a 32-bit integer, most significant octet first
       *   (network standard byte order), in centimeters, from a base of
       *   100,000m below the [WGS 84] reference spheroid used by GPS
       *   (semimajor axis a=6378137.0, reciprocal flattening
       *   rf=298.257223563).
       *   Altitude above (or below) sea level may be used as an approximation
       *   of altitude relative to the the [WGS 84] spheroid, though due to the
       *   Earth's surface not being a perfect spheroid, there will be
       *   differences.  (For example, the geoid (which sea level approximates)
       *   for the continental US ranges from 10 meters to 50 meters below the
       *   [WGS 84] spheroid.  Adjustments to ALTITUDE and/or VERT PRE will be
       *   necessary in most cases.  The Defense Mapping Agency publishes geoid
       *   height values relative to the [WGS 84] ellipsoid.
       */
      //----------------------------------------------------------------------
      uint32_t Altitude(uint32_t altitude);

      //----------------------------------------------------------------------
      //!  Encodes the LOC resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded LOC resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded LOC resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the LOC resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  LOC resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded LOC resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the LOC resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataLOC & loc);
      
    private:
      typedef struct __attribute__((packed)) {
        uint8_t   version;
        uint8_t   size;
        uint8_t   horizontalPrecision;
        uint8_t   verticalPrecision;
        uint32_t  latitude;
        uint32_t  longitude;
        uint32_t  altitude;
      } data_t;

      data_t  _data;

      static data_t ToNetworkByteOrder(const data_t *data);
      static data_t ToHostByteOrder(const data_t *data);
    };
    
  }  // namespace Dns
  
}  // namespace Dwm

#endif  // _DWMDNSRRDATALOC_HH_
