//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/tags/DwmDns-0.1.0/classes/include/DwmDnsRRDataDS.hh 10129 $
// @(#) $Id: DwmDnsRRDataDS.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataDS.hh
//!  \brief Dwm::Dns::RRDataDS class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATADS_HH_
#define _DWMDNSRRDATADS_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates DS resource record data.  See RFC 4034 for details.
    //------------------------------------------------------------------------
    class RRDataDS
      : public RRData
    {
    public:
      static const uint16_t k_rrtype = 43;
      
      static const uint8_t  k_digestTypeReserved  = 0;
      static const uint8_t  k_digestTypeSHA1      = 1;
      
      static const uint8_t  k_algorithmReserved    = 0;
      static const uint8_t  k_algorithmRSAMD5      = 1;    // not recommended
      static const uint8_t  k_algorithmDH          = 2;
      static const uint8_t  k_algorithmDSA         = 3;    // optional
      static const uint8_t  k_algorithmECC         = 4;
      static const uint8_t  k_algorithmRSASHA1     = 5;    // mandatory
      static const uint8_t  k_algorithmINDIRECT    = 252;
      static const uint8_t  k_algorithmPRIVATEDNS  = 253;  // optional
      static const uint8_t  k_algorithmPRIVATEOID  = 254;  // optional
      
      //----------------------------------------------------------------------
      //!  Constructs an empty DS resource record data object.
      //----------------------------------------------------------------------
      RRDataDS();
      
      //----------------------------------------------------------------------
      //!  Constructs DS resource record dats with @c keyTag, @c algorithm,
      //!  @c digestType and @c digest.
      //----------------------------------------------------------------------
      RRDataDS(uint16_t keyTag, uint8_t algorithm, uint8_t digestType,
               const std::string & digest);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataDS & ds) const;
      
      //----------------------------------------------------------------------
      //!  Returns the key tag in the DS resource record data.
      //----------------------------------------------------------------------
      uint16_t KeyTag() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the key tag in the DS resource record data.
      //----------------------------------------------------------------------
      uint16_t KeyTag(uint16_t keyTag);
      
      //----------------------------------------------------------------------
      //!  Returns the algorithm in the DS resource record data.
      //----------------------------------------------------------------------
      uint8_t Algorithm() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the algorithm in the DS resource record data.
      //----------------------------------------------------------------------
      uint8_t Algorithm(uint8_t algorithm);
      
      //----------------------------------------------------------------------
      //!  Returns the digest type in the DS resource record data.
      //----------------------------------------------------------------------
      uint8_t DigestType() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the digest type in the DS resource record data.
      //----------------------------------------------------------------------
      uint8_t DigestType(uint16_t digestType);
      
      //----------------------------------------------------------------------
      //!  Returns the digest in the DS resource record data.
      //----------------------------------------------------------------------
      const std::string & Digest() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the digest in the DS resource record data.
      //----------------------------------------------------------------------
      const std::string & Digest(const std::string & digest);
      
      //----------------------------------------------------------------------
      //!  Encodes the DS resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded DS resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded DS resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the DS resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  DS resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded DS resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);
      
      //----------------------------------------------------------------------
      //!  Print the DS resource record data to an ostream in human readable
      //!  form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataDS & ds);
      
    private:
      uint16_t     _keyTag;
      uint8_t      _algorithm;
      uint8_t      _digestType;
      std::string  _digest;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATADS_HH_
