//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmDnsEtcHosts.hh 10136 2018-01-28 06:01:09Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsEtcHosts.hh
//!  \brief Dwm::Dns::EtcHosts class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSETCHOSTS_HH_
#define _DWMDNSETCHOSTS_HH_

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
}

#include <cstdint>
#include <map>
#include <string>
#include <vector>

#include "DwmDnsResourceRecord.hh"
#include "DwmDnsUtils.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup resolvergroup
    //!  Encapsulates a resolver 'hosts' file, such as /etc/hosts.
    //------------------------------------------------------------------------
    class EtcHosts
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct from a path to the hosts file.  This will parse the
      //!  entries from the given @c hostsFile and populate the encapsulated
      //!  maps.
      //----------------------------------------------------------------------
      EtcHosts(const std::string & hostsFile = "/etc/hosts");
      
      //----------------------------------------------------------------------
      //!  Gets DNS resource records of type @c rrtype for the given host
      //!  @c name.  Places the resource records in @c results and returns
      //!  true on success.  Returns false on failure.
      //----------------------------------------------------------------------
      bool Get(const std::string & name, uint16_t rrtype,
               std::vector<ResourceRecord> & results);

      //----------------------------------------------------------------------
      //!  Adds an entry for the given IP address @c ipAddr named @c name.
      //!  This is only used by the parser and not normally called from
      //!  application code.  Note that we never attempt to write to the
      //!  hosts file, so an entry added vias this method is never saved
      //!  to the hosts file.
      //----------------------------------------------------------------------
      void Add(const std::string & ipAddr, const std::string & name);
      
    private:
      std::map<std::string,in_addr>          _namesToV4Addrs;
      std::map<std::string,in6_addr>         _namesToV6Addrs;
      std::multimap<in6_addr,std::string>    _v6addrsToNames;
      std::multimap<in_addr,std::string>     _v4addrsToNames;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSETCHOSTS_HH_
